# coding=utf-8
import threading
import time
import logging
import json
import uuid
import random
import datetime

from django.http import parse_cookie
from django.conf import settings
from django.utils.http import cookie_date
from django.contrib.sessions.middleware import SessionMiddleware

from gm_types.gaia import PLATFORM_CHOICES

from libs.exception_info import get_exception_info, get_func_args

exception_logger = logging.getLogger('exception_logger')
profile_logger = logging.getLogger('profile_logger')


class TrackingMiddleware(object):
    """set/get tracking cookie."""

    def _parse_xcx_cookies(self, http_cookies):
        """小程序cookie解析还原。

        由于小程序头部cookie是小写：cookie，http后端set-cookie到浏览器会响应为Cookie，
        造成小程序上传两个cookie导致直接拼接，sessionid变成不可用状态。比如小程序：
        cookie:a=b Cookie:c=d 到后端就变成了a='b,c=d'。
        """
        _cookies = {}

        for name, v in http_cookies.items():

            error_cookie = '{}={}'.format(name, v)
            raw_cookie = error_cookie.replace(',', ';')

            cookies = parse_cookie(raw_cookie)
            for k, v in cookies.items():
                _cookies[k] = v

        return _cookies

    def process_request(self, request):
        """check if tracking cookie exist"""

        if request.GET.get("platform", '') == PLATFORM_CHOICES.WECHATXCX:
            cookies = self._parse_xcx_cookies(request.COOKIES)
            for name, v in cookies.items():
                request.COOKIES[name] = v

        if settings.TRACKING_COOKIE_NAME in request.COOKIES:
            request._has_tracking_cookie = True
            request.tracking_cookie = request.COOKIES[settings.TRACKING_COOKIE_NAME]

        else:
            request._has_tracking_cookie = False
            request.tracking_cookie = uuid.uuid1().hex + str(random.randrange(1, 10000))

    def process_response(self, request, response):

        if not request._has_tracking_cookie:
            expires_time = time.time() + settings.TRACKING_COOKIE_AGE
            expires = cookie_date(expires_time)
            response.set_cookie(
                settings.TRACKING_COOKIE_NAME,
                request.tracking_cookie,
                max_age=settings.TRACKING_COOKIE_AGE,
                expires=expires,
            )

        return response


class LoggingMiddleware(object):

    def process_request(self, request):
        request.start_time = time.time()

    def process_response(self, request, response):
        execute_time = time.time() - request.start_time
        path = request.get_full_path()
        gm_request_id = request.META.get('HTTP_X_GM_REQUEST_ID')

        try:
            items = {}
            if request.method == 'GET':
                items = request.GET.dict()

            elif request.method == 'POST':
                items = request.POST.dict()

            items = json.dumps(items)
        except:
            items = ''

        profile_logger.info(
            json.dumps(
                {
                    'timestamp': datetime.datetime.fromtimestamp(
                        request.start_time
                    ).strftime('%Y-%m-%dT%H:%M:%S+08:00'),
                    'request_method': request.method,
                    'path': path,
                    'execute_time': execute_time,
                    'items': items,
                    'gm_request_id': gm_request_id
                }
                , ensure_ascii=True, encoding='utf-8'
            )
        )
        return response

    def process_exception(self, request, exception):
        gm_request_id = request.META.get('HTTP_X_GM_REQUEST_ID')
        exc = get_exception_info()
        func_args = get_func_args()
        exception_logger.info(
            json.dumps(
                {
                    'gm_request_id': gm_request_id,
                    'stacktrace': exc,
                    'func_args': func_args,
                    'timestamp': datetime.datetime.now().strftime('%Y-%m-%dT%H:%M:%S+08:00'),
                }
                , ensure_ascii=True, encoding='utf-8'
            )
        )


class TrackingCityMiddleware(object):
    def process_request(self, request):
        current_city_id = request.GET.get('current_city_id', '')
        request.logger.app(tracking_id=request.tracking_cookie, current_city_id=current_city_id)
