package com.gmei.data.dqmp.service.impl;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.gmei.data.dqmp.common.Constants;
import com.gmei.data.dqmp.domain.TblResultCheckRefer;
import com.gmei.data.dqmp.domain.TblResultCheckUnblank;
import com.gmei.data.dqmp.domain.TblResultCheckUnique;
import com.gmei.data.dqmp.domain.TblSqlCheckDuplex;
import com.gmei.data.dqmp.domain.TblSqlCheckDuplexCriteria;
import com.gmei.data.dqmp.domain.TblSqlCheckSingle;
import com.gmei.data.dqmp.domain.TblSqlCheckSingleCriteria;
import com.gmei.data.dqmp.mapper.TblResultCheckReferMapper;
import com.gmei.data.dqmp.mapper.TblResultCheckUnblankMapper;
import com.gmei.data.dqmp.mapper.TblResultCheckUniqueMapper;
import com.gmei.data.dqmp.mapper.TblSqlCheckDuplexMapper;
import com.gmei.data.dqmp.mapper.TblSqlCheckSingleMapper;
import com.gmei.data.dqmp.pool.JdbcConnectPool;
import com.gmei.data.dqmp.service.CheckService;
import com.gmei.data.dqmp.service.DingdingService;
import com.gmei.data.dqmp.service.MailService;
import com.gmei.data.dqmp.utils.DateUtils;

@Service
public class CheckServiceImpl implements CheckService {

	@Autowired
	private TblSqlCheckSingleMapper tblSqlCheckSingleMapper;
	@Autowired
	private TblSqlCheckDuplexMapper tblSqlCheckDuplexMapper;
	@Autowired
	private TblResultCheckUniqueMapper tblResultCheckUniqueMapper;
	@Autowired
	private TblResultCheckUnblankMapper tblResultCheckUnblankMapper;
	@Autowired
	private TblResultCheckReferMapper tblResultCheckReferMapper;
	@Autowired
	private MailService mailService;
	@Autowired
	private DingdingService dingdingService;
	private static final Logger logger = LoggerFactory.getLogger(CheckServiceImpl.class);

	/**
	 * 校验并持久化简单型校验结果
	 * @param checkType
	 */
	@Override
	public void checkAndPersistSimpleResult(String checkType) {
		TblSqlCheckSingleCriteria tblCheckSingleSqlCriteria = new TblSqlCheckSingleCriteria();
		tblCheckSingleSqlCriteria.createCriteria()
			.andCheckTypeEqualTo(checkType)
			.andIsValidEqualTo(Constants.IS_VALID_ON);
		List<TblSqlCheckSingle> sqlList = tblSqlCheckSingleMapper.selectByExampleWithBLOBs(tblCheckSingleSqlCriteria);
		for (TblSqlCheckSingle tblCheckSql : sqlList) {
			String sql = tblCheckSql.getSqlContent();
			Integer id = tblCheckSql.getId();
			Double threshold = tblCheckSql.getThreshold();
			int rs = 0;
			if (Constants.CHECK_UNIQUE.equals(tblCheckSql.getCheckType())) {
				TblResultCheckUnique uniqueCheckResult = getUniqueCheckResult(sql);
				if (uniqueCheckResult == null) {
					logger.error("Find result is empty!");
					return;
				}else {
					Double uniqueRate = uniqueCheckResult.getUniqueRate();
					if(uniqueRate <= threshold) {
						mailService.sendSimpleMail(uniqueCheckResult.toString());
						dingdingService.sendMsgToDingding(uniqueCheckResult.toString());
					}
				}
				uniqueCheckResult.setSqlId(id);
				rs = tblResultCheckUniqueMapper.insert(uniqueCheckResult);
			} else if (Constants.CHECK_UNBLANK.equals(tblCheckSql.getCheckType())){
				TblResultCheckUnblank tblResultCheckUnblank = getUnblankCheckResult(sql);
				if (tblResultCheckUnblank == null) {
					logger.error("Find result is empty!");
					return;
				}else {
					Double unblankRate = tblResultCheckUnblank.getUnblankRate();
					if(unblankRate <= threshold) {
						mailService.sendSimpleMail(tblResultCheckUnblank.toString());
						dingdingService.sendMsgToDingding(tblResultCheckUnblank.toString());
					}
				}
				tblResultCheckUnblank.setSqlId(id);
				rs = tblResultCheckUnblankMapper.insert(tblResultCheckUnblank);
			}
			if (rs == 1) {
				logger.info("Run success: {}", sql);
			} else {
				logger.error("Run failed: {}", sql);
			}
		}
	}
	
	/**
	 * 校验并持久化复杂型校验结果
	 * @param checkType
	 */
	@Override
	public void checkAndPersistMultipleResult(String checkType) {
		TblSqlCheckDuplexCriteria tblCheckMultipleSqlCriteria = new TblSqlCheckDuplexCriteria();
		tblCheckMultipleSqlCriteria.createCriteria()
			.andCheckTypeEqualTo(checkType)
			.andIsValidEqualTo(Constants.IS_VALID_ON);
		List<TblSqlCheckDuplex> sqlList = tblSqlCheckDuplexMapper.selectByExampleWithBLOBs(tblCheckMultipleSqlCriteria);
		for (TblSqlCheckDuplex tblCheckSql : sqlList) {
			String sql = tblCheckSql.getSqlContent();
			Integer id = tblCheckSql.getId();
			Double threshold = tblCheckSql.getThreshold();
			int rs = 0;
			if (Constants.CHECK_REFER.equals(tblCheckSql.getCheckType())) {
				TblResultCheckRefer referCheckResult = getReferCheckResult(sql);
				if (referCheckResult == null) {
					logger.error("Find result is empty!");
					return;
				}else {
					Double matchedRate = referCheckResult.getMatchedRate();
					if(matchedRate <= threshold) {
						mailService.sendSimpleMail(referCheckResult.toString());
						dingdingService.sendMsgToDingding(referCheckResult.toString());
					}
				}
				referCheckResult.setSqlId(id);
				rs = tblResultCheckReferMapper.insert(referCheckResult);
			}
			if (rs == 1) {
				logger.info("Run success: {}", sql);
			} else {
				logger.error("Run failed: {}", sql);
			}
		}
	}

	/**
	 * 获取唯一性校验结果
	 * @param sql
	 * @return
	 */
	private TblResultCheckUnique getUniqueCheckResult(String sql) {
		sql = sql.replaceAll("#partation_date", String.format("'%s'", DateUtils.getYesterdayDateStr()));
		logger.info("Sql content : {}", sql);
		TblResultCheckUnique tblResultCheckUnique = null;
		if (StringUtils.isBlank(sql)) {
			return tblResultCheckUnique;
		}
		try {
			Connection conn = JdbcConnectPool.getConnect();
			PreparedStatement ps = conn.prepareStatement(sql);
			ResultSet rs = ps.executeQuery();
			while (rs.next()) {
				tblResultCheckUnique = new TblResultCheckUnique();
				tblResultCheckUnique.setDbName(rs.getString("db_name"));
				tblResultCheckUnique.setTbName(rs.getString("tb_name"));
				tblResultCheckUnique.setColName(rs.getString("col_name"));
				tblResultCheckUnique.setAllNum(rs.getLong("all_num"));
				tblResultCheckUnique.setUniqueNum(rs.getLong("unique_num"));
				tblResultCheckUnique.setUniqueRate(rs.getDouble("unique_rate"));
				tblResultCheckUnique.setPartitionDate(rs.getString("partition_date"));
				tblResultCheckUnique.setCreateTime(rs.getString("create_time"));
			}
			rs.close();
			ps.close();
			JdbcConnectPool.releaseConnection(conn);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return tblResultCheckUnique;
	}

	/**
	 * 获取非空性校验结果
	 * @param sql
	 * @return
	 */
	private TblResultCheckUnblank getUnblankCheckResult(String sql) {
		sql = sql.replaceAll("#partation_date", String.format("'%s'", DateUtils.getYesterdayDateStr()));
		logger.info("Sql content : {}", sql);
		TblResultCheckUnblank tblResultCheckUnblank = null;
		if (StringUtils.isBlank(sql)) {
			return tblResultCheckUnblank;
		}
		try {
			Connection conn = JdbcConnectPool.getConnect();
			PreparedStatement ps = conn.prepareStatement(sql);
			ResultSet rs = ps.executeQuery();
			while (rs.next()) {
				tblResultCheckUnblank = new TblResultCheckUnblank();
				tblResultCheckUnblank.setDbName(rs.getString("db_name"));
				tblResultCheckUnblank.setTbName(rs.getString("tb_name"));
				tblResultCheckUnblank.setColName(rs.getString("col_name"));
				tblResultCheckUnblank.setAllNum(rs.getLong("all_num"));
				tblResultCheckUnblank.setUnblankNum(rs.getLong("unblank_num"));
				tblResultCheckUnblank.setUnblankRate(rs.getDouble("unblank_rate"));
				tblResultCheckUnblank.setPartitionDate(rs.getString("partition_date"));
				tblResultCheckUnblank.setCreateTime(rs.getString("create_time"));
			}
			rs.close();
			ps.close();
			JdbcConnectPool.releaseConnection(conn);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return tblResultCheckUnblank;
	}
	
	/**
	 * 获取参照性校验结果
	 * @param sql
	 * @return
	 */
	private TblResultCheckRefer getReferCheckResult(String sql) {
		sql = sql.replaceAll("#partation_date", String.format("'%s'", DateUtils.getYesterdayDateStr()));
		logger.info("Sql content : {}", sql);
		TblResultCheckRefer tblResultCheckRefer = null;
		if (StringUtils.isBlank(sql)) {
			return tblResultCheckRefer;
		}
		try {
			Connection conn = JdbcConnectPool.getConnect();
			PreparedStatement ps = conn.prepareStatement(sql);
			ResultSet rs = ps.executeQuery();
			while (rs.next()) {
				tblResultCheckRefer = new TblResultCheckRefer();
				tblResultCheckRefer.setCheckDbName(rs.getString("check_db_name"));
				tblResultCheckRefer.setCheckTbName(rs.getString("check_tb_name"));
				tblResultCheckRefer.setCheckColName(rs.getString("check_col_name"));
				tblResultCheckRefer.setReferDbName(rs.getString("refer_db_name"));
				tblResultCheckRefer.setReferTbName(rs.getString("refer_tb_name"));
				tblResultCheckRefer.setReferColName(rs.getString("refer_col_name"));
				tblResultCheckRefer.setAllNum(rs.getLong("all_num"));
				tblResultCheckRefer.setMatchedNum(rs.getLong("matched_num"));
				tblResultCheckRefer.setMatchedRate(rs.getDouble("matched_rate"));
				tblResultCheckRefer.setPartitionDate(rs.getString("partition_date"));
				tblResultCheckRefer.setCreateTime(rs.getString("create_time"));
			}
			rs.close();
			ps.close();
			JdbcConnectPool.releaseConnection(conn);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return tblResultCheckRefer;
	}
	
	public static void main(String[] args) {
	}
}
