package com.gmei.data.dqmp.service.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.gmei.data.dqmp.common.Constants;
import com.gmei.data.dqmp.domain.TblIndicatorOperatorInfo;
import com.gmei.data.dqmp.domain.TblIndicatorOperatorInfoCriteria;
import com.gmei.data.dqmp.domain.TblRuleInfoDuplexCriteria;
import com.gmei.data.dqmp.domain.TblRuleInfoDuplexWithBLOBs;
import com.gmei.data.dqmp.domain.TblRuleInfoSimple;
import com.gmei.data.dqmp.domain.TblRuleInfoSimpleCriteria;
import com.gmei.data.dqmp.domain.TblRuleTemplate;
import com.gmei.data.dqmp.domain.TblRuleTemplateCriteria;
import com.gmei.data.dqmp.domain.TblSqlCheckDuplex;
import com.gmei.data.dqmp.domain.TblSqlCheckDuplexCriteria;
import com.gmei.data.dqmp.domain.TblSqlCheckSingle;
import com.gmei.data.dqmp.domain.TblSqlCheckSingleCriteria;
import com.gmei.data.dqmp.domain.TblSqlMonitorSpecial;
import com.gmei.data.dqmp.domain.TblSqlMonitorSpecialCriteria;
import com.gmei.data.dqmp.domain.TblSqlMonitorVolatility;
import com.gmei.data.dqmp.domain.TblSqlMonitorVolatilityCriteria;
import com.gmei.data.dqmp.mapper.TblIndicatorOperatorInfoMapper;
import com.gmei.data.dqmp.mapper.TblRuleInfoDuplexMapper;
import com.gmei.data.dqmp.mapper.TblRuleInfoSimpleMapper;
import com.gmei.data.dqmp.mapper.TblRuleTemplateMapper;
import com.gmei.data.dqmp.mapper.TblSqlCheckDuplexMapper;
import com.gmei.data.dqmp.mapper.TblSqlCheckSingleMapper;
import com.gmei.data.dqmp.mapper.TblSqlMonitorSpecialMapper;
import com.gmei.data.dqmp.mapper.TblSqlMonitorVolatilityMapper;
import com.gmei.data.dqmp.service.DqRuleService;
import com.gmei.data.dqmp.utils.BeanUtils;
import com.gmei.data.dqmp.utils.DateUtils;
import com.gmei.data.dqmp.vo.DqReferRuleVo;
import com.gmei.data.dqmp.vo.DqRuleVo;
import com.gmei.data.dqmp.vo.DqSpecialRuleVo;
import com.gmei.data.dqmp.vo.DqUnblankRuleVo;
import com.gmei.data.dqmp.vo.DqUniqueRuleVo;
import com.gmei.data.dqmp.vo.DqVolatilityRuleVo;

@Service
public class DqRuleServiceImpl implements DqRuleService {

	@Autowired
	private TblRuleInfoSimpleMapper tblRuleInfoSimpleMapper;
	@Autowired
	private TblRuleInfoDuplexMapper tblRuleInfoDuplexMapper;
	@Autowired
	private TblSqlCheckSingleMapper tblSqlCheckSingleMapper;
	@Autowired
	private TblSqlCheckDuplexMapper tblSqlCheckMultipleMapper;
	@Autowired
	private TblSqlMonitorVolatilityMapper tblSqlMonitorVolatilityMapper;
	@Autowired
	private TblSqlMonitorSpecialMapper tblSqlMonitorSpecialMapper;
	@Autowired
	private TblRuleTemplateMapper tblRuleTemplateMapper;
	@Autowired
	private TblIndicatorOperatorInfoMapper tblIndicatorOperatorInfoMapper;
	private static final Logger logger = LoggerFactory.getLogger(DqRuleServiceImpl.class);

	@Override
	@Transactional
	public Boolean addDqRule(DqRuleVo dqRuleVo) {
		Boolean rs = false;
		String checkType = dqRuleVo.getCheckType();
		if (Constants.CHECK_UNIQUE.equals(checkType) || Constants.CHECK_UNBLANK.equals(checkType)
				|| Constants.CHECK_VOLATILITY.equals(checkType) || Constants.CHECK_SPECIAL.equals(checkType)) {
			rs = simpleRuleInfoInsert(dqRuleVo);
		} else if (Constants.CHECK_REFER.equals(checkType)) {
			rs = duplexRuleInfoInsert(dqRuleVo);
		} else {
			logger.error("checkType is illegal! {}", checkType);
			return rs;
		}
		return rs;
	}

	/**
	 * 简单校验规则新增
	 * 
	 * @param dqRuleVo
	 * @return
	 */
	private Boolean simpleRuleInfoInsert(DqRuleVo dqRuleVo) {
		if (!simpleRuleInfoCheck(dqRuleVo)) {
			logger.error("Param is illegal! {}", dqRuleVo);
			return false;
		}
		TblRuleInfoSimple record = BeanUtils.map(dqRuleVo, TblRuleInfoSimple.class);
		record.setStartTime(DateUtils.getZeroTimeStrsMap().get(record.getStartTime()));
		int rs = tblRuleInfoSimpleMapper.insertSelective(record);
		if (rs == 1) {
			String checkType = dqRuleVo.getCheckType();
			switch (checkType) {
			case Constants.CHECK_UNIQUE:
				TblSqlCheckSingle tblSqlCheckSingle = BeanUtils.map(dqRuleVo, TblSqlCheckSingle.class);
				tblSqlCheckSingle.setSqlContent(genSqlByTemplate(dqRuleVo));
				tblSqlCheckSingle.setParentId(record.getId());
				tblSqlCheckSingleMapper.insertSelective(tblSqlCheckSingle);
				break;
			case Constants.CHECK_UNBLANK:
				TblSqlCheckSingle tcss = BeanUtils.map(dqRuleVo, TblSqlCheckSingle.class);
				tcss.setSqlContent(genSqlByTemplate(dqRuleVo));
				tcss.setParentId(record.getId());
				tblSqlCheckSingleMapper.insertSelective(tcss);
				break;
			case Constants.CHECK_VOLATILITY:
				if (!volatilityRuleParamsCheck(dqRuleVo)) {
					logger.error("Param is illegal! {}", dqRuleVo);
					return false;
				}
				TblSqlMonitorVolatility tblSqlMonitorVolatility = BeanUtils.map(dqRuleVo,
						TblSqlMonitorVolatility.class);
				tblSqlMonitorVolatility.setSqlContent(genSqlByTemplate(dqRuleVo));
				tblSqlMonitorVolatility.setParentId(record.getId());
				tblSqlMonitorVolatilityMapper.insertSelective(tblSqlMonitorVolatility);
				break;
			case Constants.CHECK_SPECIAL:
				if (!specialRuleParamsCheck(dqRuleVo)) {
					logger.error("Param is illegal! {}", dqRuleVo);
					return false;
				}
				dqRuleVo.setStartTime(DateUtils.getZeroTimeStrsMap().get(dqRuleVo.getStartTime()));
				TblSqlMonitorSpecial tblSqlMonitorSpecial = BeanUtils.map(dqRuleVo, TblSqlMonitorSpecial.class);
				tblSqlMonitorSpecial.setSqlContent(genSqlByTemplate(dqRuleVo));
				tblSqlMonitorSpecial.setParentId(record.getId());
				tblSqlMonitorSpecialMapper.insertSelective(tblSqlMonitorSpecial);
				break;
			default:
				logger.error("checkType value is error! {}", checkType);
				return false;
			}
		}
		return true;
	}

	/**
	 * 双表校验规则录入
	 * 
	 * @param dqRuleVo
	 * @return
	 */
	private Boolean duplexRuleInfoInsert(DqRuleVo dqRuleVo) {
		if (!duplexRuleInfoCheck(dqRuleVo)) {
			logger.error("Param is illegal! {}", dqRuleVo);
			return false;
		}
		TblRuleInfoDuplexWithBLOBs record = BeanUtils.map(dqRuleVo, TblRuleInfoDuplexWithBLOBs.class);
		int rs = tblRuleInfoDuplexMapper.insertSelective(record);
		if (rs == 1) {
			String checkType = dqRuleVo.getCheckType();
			if (Constants.CHECK_REFER.equals(checkType)) {
				TblSqlCheckDuplex tblSqlCheckDuplex = BeanUtils.map(dqRuleVo, TblSqlCheckDuplex.class);
				String sqlContent = genSqlByTemplate(dqRuleVo);
				tblSqlCheckDuplex.setSqlContent(sqlContent);
				tblSqlCheckDuplex.setParentId(record.getId());
				tblSqlCheckMultipleMapper.insertSelective(tblSqlCheckDuplex);
			} else {
				logger.error("checkType value is error! {}", checkType);
				return false;
			}
		}
		return true;
	}

	private boolean simpleRuleInfoCheck(DqRuleVo dqRuleVo) {
		if (StringUtils.isBlank(dqRuleVo.getDbName()) || StringUtils.isBlank(dqRuleVo.getTbName())) {
			return false;
		}
		if (Constants.CHECK_SPECIAL.equals(dqRuleVo.getCheckType())) {
			if (StringUtils.isBlank(dqRuleVo.getTimeColName())) {
				return false;
			}
		} else {
			if (StringUtils.isBlank(dqRuleVo.getColName())) {
				return false;
			}
		}
		return true;
	}

	private boolean volatilityRuleParamsCheck(DqRuleVo dqRuleVo) {
		if (StringUtils.isBlank(dqRuleVo.getIndicatorType())) {
			return false;
		}
		return true;
	}

	private boolean specialRuleParamsCheck(DqRuleVo dqRuleVo) {
		if (StringUtils.isBlank(dqRuleVo.getTimeColName()) || StringUtils.isBlank(dqRuleVo.getStartTime())) {
			return false;
		}
		return true;
	}

	private boolean duplexRuleInfoCheck(DqRuleVo dqRuleVo) {
		if (StringUtils.isBlank(dqRuleVo.getCheckDbName()) || StringUtils.isBlank(dqRuleVo.getCheckTbName())
				|| StringUtils.isBlank(dqRuleVo.getCheckColName()) || StringUtils.isBlank(dqRuleVo.getReferColName())
				|| StringUtils.isBlank(dqRuleVo.getReferDbName()) || StringUtils.isBlank(dqRuleVo.getReferTbName())) {
			return false;
		}
		return true;
	}

	private String genSqlByTemplate(DqRuleVo dqRuleVo) {
		TblRuleTemplateCriteria example = new TblRuleTemplateCriteria();
		example.createCriteria().andCheckTypeEqualTo(dqRuleVo.getCheckType());
		TblRuleTemplate tblRuleTemplate = tblRuleTemplateMapper.selectByExampleWithBLOBs(example).get(0);
		String tpltContent = tblRuleTemplate.getTpltContent();
		Map<String, String> indicatorMap = new HashMap<String, String>();
		TblIndicatorOperatorInfoCriteria tblIndicatorOperatorInfoCriteria = new TblIndicatorOperatorInfoCriteria();
		List<TblIndicatorOperatorInfo> tblIndicatorOperatorInfos = tblIndicatorOperatorInfoMapper
				.selectByExample(tblIndicatorOperatorInfoCriteria);
		for (TblIndicatorOperatorInfo tblIndicatorOperatorInfo : tblIndicatorOperatorInfos) {
			indicatorMap.put(tblIndicatorOperatorInfo.getIndicatorType(), tblIndicatorOperatorInfo.getOperatorExpr());
		}
		String indicatorType = dqRuleVo.getIndicatorType();
		String rs = tpltContent.replaceAll("#db_name", dqRuleVo.getDbName())
				.replaceAll("#tb_name", dqRuleVo.getTbName()).replaceAll("#indicator_type", indicatorType)
				.replaceAll("#indicator_expre", indicatorMap.get(indicatorType))
				.replaceAll("#col_name", dqRuleVo.getColName())
				.replaceAll("#start_time", "'" + dqRuleVo.getStartTime() + "'")
				.replaceAll("#end_time", "'" + DateUtils.getTodayZeroTimeStr() + "'")
				.replaceAll("#check_db_name", dqRuleVo.getCheckDbName())
				.replaceAll("#check_tb_name", dqRuleVo.getCheckTbName())
				.replaceAll("#check_col_name", dqRuleVo.getCheckColName())
				.replaceAll("#refer_db_name", dqRuleVo.getReferDbName())
				.replaceAll("#refer_tb_name", dqRuleVo.getReferTbName())
				.replaceAll("#refer_col_name", dqRuleVo.getReferColName())
				.replaceAll("#time_col_name", dqRuleVo.getTimeColName());

		String filters = dqRuleVo.getFilters();
		if (null != filters) {
			filters = filters.replaceAll("\\$", "RDS_CHAR_DOLLAR");
			rs = rs.replaceAll("#filters", filters);
		}
		String checkFilters = dqRuleVo.getCheckFilters();
		if (null != checkFilters) {
			checkFilters = checkFilters.replaceAll("\\$", "RDS_CHAR_DOLLAR");
			rs = rs.replaceAll("#check_filters", checkFilters);
		}
		String referFilters = dqRuleVo.getReferFilters();
		if (null != referFilters) {
			referFilters = referFilters.replaceAll("\\$", "RDS_CHAR_DOLLAR");
			rs = rs.replaceAll("#refer_filters", referFilters);
		}
		rs = rs.replaceAll("RDS_CHAR_DOLLAR", "\\$");
		return rs;
	}

	@Override
	public List<DqUniqueRuleVo> findUniqueDqRuleList() {
		List<DqUniqueRuleVo> rsList = new ArrayList<DqUniqueRuleVo>();
		TblRuleInfoSimpleCriteria example = new TblRuleInfoSimpleCriteria();
		example.createCriteria().andCheckTypeEqualTo(Constants.CHECK_UNIQUE);
		List<TblRuleInfoSimple> selectByExample = tblRuleInfoSimpleMapper.selectByExampleWithBLOBs(example);
		for (TblRuleInfoSimple tblRuleInfoSimple : selectByExample) {
			DqUniqueRuleVo dqUniqueRuleVo = BeanUtils.map(tblRuleInfoSimple, DqUniqueRuleVo.class);
			if (tblRuleInfoSimple.getIsValid() == Constants.IS_VALID_ON) {
				dqUniqueRuleVo.setStatus("正常");
			} else {
				dqUniqueRuleVo.setStatus("禁用");
			}
			rsList.add(dqUniqueRuleVo);
		}
		return rsList;
	}

	@Override
	public List<DqUnblankRuleVo> findUnblankDqRuleList() {
		List<DqUnblankRuleVo> rsList = new ArrayList<DqUnblankRuleVo>();
		TblRuleInfoSimpleCriteria example = new TblRuleInfoSimpleCriteria();
		example.createCriteria().andCheckTypeEqualTo(Constants.CHECK_UNBLANK);
		List<TblRuleInfoSimple> selectByExample = tblRuleInfoSimpleMapper.selectByExampleWithBLOBs(example);
		for (TblRuleInfoSimple tblRuleInfoSimple : selectByExample) {
			DqUnblankRuleVo dqUnblankRuleVo = BeanUtils.map(tblRuleInfoSimple, DqUnblankRuleVo.class);
			if (tblRuleInfoSimple.getIsValid() == Constants.IS_VALID_ON) {
				dqUnblankRuleVo.setStatus("正常");
			} else {
				dqUnblankRuleVo.setStatus("禁用");
			}
			rsList.add(dqUnblankRuleVo);
		}
		return rsList;
	}

	@Override
	public List<DqReferRuleVo> findReferDqRuleList() {
		List<DqReferRuleVo> rsList = new ArrayList<DqReferRuleVo>();
		TblRuleInfoDuplexCriteria example = new TblRuleInfoDuplexCriteria();
		example.createCriteria().andCheckTypeEqualTo(Constants.CHECK_REFER);
		List<TblRuleInfoDuplexWithBLOBs> selectByExampleWithBLOBs = tblRuleInfoDuplexMapper
				.selectByExampleWithBLOBs(example);
		for (TblRuleInfoDuplexWithBLOBs tblRuleInfoDuplexWithBLOBs : selectByExampleWithBLOBs) {
			DqReferRuleVo dqReferRuleVo = BeanUtils.map(tblRuleInfoDuplexWithBLOBs, DqReferRuleVo.class);
			if (tblRuleInfoDuplexWithBLOBs.getIsValid() == Constants.IS_VALID_ON) {
				dqReferRuleVo.setStatus("正常");
			} else {
				dqReferRuleVo.setStatus("禁用");
			}
			rsList.add(dqReferRuleVo);
		}
		return rsList;
	}

	@Override
	public List<DqVolatilityRuleVo> findVolatilityDqRuleList() {
		List<DqVolatilityRuleVo> rsList = new ArrayList<DqVolatilityRuleVo>();
		TblRuleInfoSimpleCriteria example = new TblRuleInfoSimpleCriteria();
		example.createCriteria().andCheckTypeEqualTo(Constants.CHECK_VOLATILITY);
		List<TblRuleInfoSimple> selectByExample = tblRuleInfoSimpleMapper.selectByExampleWithBLOBs(example);
		for (TblRuleInfoSimple tblRuleInfoSimple : selectByExample) {
			DqVolatilityRuleVo dqVolatilityRuleVo = BeanUtils.map(tblRuleInfoSimple, DqVolatilityRuleVo.class);
			if (tblRuleInfoSimple.getIsValid() == Constants.IS_VALID_ON) {
				dqVolatilityRuleVo.setStatus("正常");
			} else {
				dqVolatilityRuleVo.setStatus("禁用");
			}
			rsList.add(dqVolatilityRuleVo);
		}
		return rsList;
	}

	@Override
	public List<DqSpecialRuleVo> findSpecialDqRuleList() {
		List<DqSpecialRuleVo> rsList = new ArrayList<DqSpecialRuleVo>();
		TblRuleInfoSimpleCriteria example = new TblRuleInfoSimpleCriteria();
		example.createCriteria().andCheckTypeEqualTo(Constants.CHECK_SPECIAL);
		List<TblRuleInfoSimple> selectByExample = tblRuleInfoSimpleMapper.selectByExampleWithBLOBs(example);
		for (TblRuleInfoSimple tblRuleInfoSimple : selectByExample) {
			DqSpecialRuleVo dqSpecialRuleVo = BeanUtils.map(tblRuleInfoSimple, DqSpecialRuleVo.class);
			if (tblRuleInfoSimple.getIsValid() == Constants.IS_VALID_ON) {
				dqSpecialRuleVo.setStatus("正常");
			} else {
				dqSpecialRuleVo.setStatus("禁用");
			}
			rsList.add(dqSpecialRuleVo);
		}
		return rsList;
	}

	@Override
	@Transactional
	public Boolean delDqRule(DqRuleVo dqRuleVo) {
		Integer id = dqRuleVo.getId();
		String checkType = dqRuleVo.getCheckType();
		if (Constants.CHECK_REFER.equals(checkType)) {
			tblRuleInfoDuplexMapper.deleteByPrimaryKey(id);
			TblSqlCheckDuplexCriteria example = new TblSqlCheckDuplexCriteria();
			example.createCriteria().andCheckTypeEqualTo(dqRuleVo.getCheckType()).andParentIdEqualTo(id);
			tblSqlCheckMultipleMapper.deleteByExample(example);
		} else {
			tblRuleInfoSimpleMapper.deleteByPrimaryKey(id);
			switch (checkType) {
			case Constants.CHECK_UNIQUE:
			case Constants.CHECK_UNBLANK:
				TblSqlCheckSingleCriteria example = new TblSqlCheckSingleCriteria();
				example.createCriteria().andCheckTypeEqualTo(checkType).andParentIdEqualTo(id);
				tblSqlCheckSingleMapper.deleteByExample(example );
				break;
			case Constants.CHECK_VOLATILITY:
				TblSqlMonitorVolatilityCriteria tblSqlMonitorVolatilityCriteria = new TblSqlMonitorVolatilityCriteria();
				tblSqlMonitorVolatilityCriteria.createCriteria().andParentIdEqualTo(id);
				tblSqlMonitorVolatilityMapper.deleteByExample(tblSqlMonitorVolatilityCriteria);
				break;
			case Constants.CHECK_SPECIAL:
				TblSqlMonitorSpecialCriteria tblSqlMonitorSpecialCriteria = new TblSqlMonitorSpecialCriteria();
				tblSqlMonitorSpecialCriteria.createCriteria().andParentIdEqualTo(id);
				tblSqlMonitorSpecialMapper.deleteByExample(tblSqlMonitorSpecialCriteria);
				break;
			default:
				break;
			}
		}
		return true;
	}
}
