package com.gmei.function;

import com.gmei.bean.bl.BlPreciseExposureBean;
import com.gmei.utils.DateUtil;
import com.google.common.hash.BloomFilter;
import com.google.common.hash.Funnels;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.streaming.api.functions.KeyedProcessFunction;
import org.apache.flink.util.Collector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.Charset;

public class BloomFilterFunction extends KeyedProcessFunction<String, BlPreciseExposureBean,BlPreciseExposureBean> {
    private static final long serialVersionUID = 1L;
    private static final Logger LOGGER = LoggerFactory.getLogger(BloomFilterFunction.class);
    private static final int BF_CARDINAL_THRESHOLD = 3000000;
    private static final double BF_FALSE_POSITIVE_RATE = 0.01;
    private volatile BloomFilter<String> blPreciseExposureBF;

    @Override
    public void onTimer(long timestamp, OnTimerContext ctx, Collector<BlPreciseExposureBean> out) throws Exception {
        super.onTimer(timestamp, ctx, out);
        long s = System.currentTimeMillis();
        blPreciseExposureBF = BloomFilter.create(Funnels.stringFunnel(Charset.forName("utf-8")), BF_CARDINAL_THRESHOLD, BF_FALSE_POSITIVE_RATE);
        long e = System.currentTimeMillis();
        LOGGER.info("Timer triggered & resetted Guava BloomFilter, time cost: " + (e - s));
    }

    @Override
    public void open(Configuration parameters) throws Exception {
        super.open(parameters);
        long s = System.currentTimeMillis();
        blPreciseExposureBF = BloomFilter.create(Funnels.stringFunnel(Charset.forName("utf-8")), BF_CARDINAL_THRESHOLD, BF_FALSE_POSITIVE_RATE);
        long e = System.currentTimeMillis();
        LOGGER.info("Created Guava BloomFilter, time cost: " + (e - s));
    }

    @Override
    public void close() throws Exception {
        super.close();
        blPreciseExposureBF = null;
    }

    @Override
    public void processElement(BlPreciseExposureBean blPreciseExposureBean, Context context, Collector<BlPreciseExposureBean> collector) throws Exception {
        String blPreciseExposureBeanId = blPreciseExposureBean.getJson();
        if (!blPreciseExposureBF.mightContain(blPreciseExposureBeanId)) {
            blPreciseExposureBF.put(blPreciseExposureBeanId);
            collector.collect(blPreciseExposureBean);
        }
        context.timerService().registerProcessingTimeTimer(DateUtil.tomorrowZeroTimestampMs(Double.valueOf(blPreciseExposureBean.getGm_nginx_timestamp()).longValue() * 1000, 8) + 1);
    }
}
