package com.gmei;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.gmei.bean.BackendEtl;
import com.gmei.map.GainValueMap;
import com.gmei.schama.GMLoggingSchema;
import com.gmei.sink.KafkaSink;
import com.gmei.utils.GmKafkaConsumer;
import com.gmei.utils.StringUtils;
import org.apache.flink.api.common.functions.FilterFunction;
import org.apache.flink.api.common.functions.MapFunction;
import org.apache.flink.api.common.restartstrategy.RestartStrategies;
import org.apache.flink.api.java.functions.KeySelector;
import org.apache.flink.api.java.utils.ParameterTool;
import org.apache.flink.runtime.state.filesystem.FsStateBackend;
import org.apache.flink.streaming.api.collector.selector.OutputSelector;
import org.apache.flink.streaming.api.datastream.DataStream;
import org.apache.flink.streaming.api.datastream.SplitStream;
import org.apache.flink.streaming.api.environment.CheckpointConfig;
import org.apache.flink.streaming.api.environment.StreamExecutionEnvironment;
import org.apache.kafka.clients.consumer.ConsumerConfig;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.ISODateTimeFormat;
import scala.Tuple4;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;

/**
 * ClassName: com.gmei.FlinkServer
 * Function: TODO ADD FUNCTION.
 * Reason: 对后端埋点进行Etl并获取活跃设备第一次活跃信息(实时)
 * Date: 2020-03-03 00:00:00
 *
 * @author sjxuwei
 * @since JDK 1.8
 */
public class FlinkServer {
    public static final String GM_INTERNAL_NON_PERSISTENT_CHANNEL = "GM-INTERNAL-NON-PERSISTENT";
    public static final DateTimeFormatter dateTimeFormat = ISODateTimeFormat.dateTime();
    public static final DateTimeFormatter dateTimeNoMillisFormat = ISODateTimeFormat.dateTimeNoMillis();
    public static void main(String[] args) throws Exception {
        String inBrokers = "localhost:9092";
        String inTopic = "test_kafka";
        String groupId = "test";
        String outJdbcUrl;
        String outTable;
        String outBrokers;
        String zxJdbcUrl;
        String outTopic;
        String startTime;
        String checkpointPath;
        Boolean isStartEarly;
        int parallelism;
        int windowSize;

        ParameterTool parameterTool = ParameterTool.fromArgs(args);
        outTable = parameterTool.get("outTable");
        inBrokers = parameterTool.get("inBrokers");
        inTopic = parameterTool.get("inTopic");
        groupId = parameterTool.get("groupId");
        zxJdbcUrl = parameterTool.get("zxJdbcUrl");
        outJdbcUrl = parameterTool.get("outJdbcUrl");
        outBrokers = parameterTool.get("outBrokers");
        outTopic = parameterTool.get("outTopic");
        startTime = parameterTool.get("startTime");
        checkpointPath = parameterTool.get("checkpointPath");
        isStartEarly = parameterTool.getBoolean("isStartEarly",false);
        parallelism = parameterTool.getInt("parallelism",1);
        windowSize = parameterTool.getInt("windowSize",10);

        System.out.println("=================params=================");
        System.out.println("outTable:" + outTable);
        System.out.println("inBrokers:" + inBrokers);
        System.out.println("inTopic:" + inTopic);
        System.out.println("groupId:" + groupId);
        System.out.println("zxJdbcUrl:" + zxJdbcUrl);
        System.out.println("outJdbcUrl:" + outJdbcUrl);
        System.out.println("outBrokers:" + outBrokers);
        System.out.println("outTopic:" + outTopic);
        System.out.println("startTime:" + startTime);
        System.out.println("checkpointPath:" + checkpointPath);
        System.out.println("isStartEarly:" + isStartEarly);
        System.out.println("parallelism:" + parallelism);
        System.out.println("windowSize:" + windowSize);
        System.out.println("========================================");

        StreamExecutionEnvironment env = StreamExecutionEnvironment.getExecutionEnvironment();

        env.enableCheckpointing(300000);
        env.setStateBackend(new FsStateBackend(checkpointPath));
        env.setRestartStrategy(RestartStrategies.fixedDelayRestart(1, 3000));
        CheckpointConfig config = env.getCheckpointConfig();
        config.enableExternalizedCheckpoints(CheckpointConfig.ExternalizedCheckpointCleanup.RETAIN_ON_CANCELLATION);
        config.setMinPauseBetweenCheckpoints(500);
        config.setCheckpointTimeout(180000);

        GmKafkaConsumer gmKafkaConsumer = new GmKafkaConsumer(inTopic);
        gmKafkaConsumer.setSource(new GMLoggingSchema());
        gmKafkaConsumer.setProp(ConsumerConfig.BOOTSTRAP_SERVERS_CONFIG,inBrokers);
        gmKafkaConsumer.setProp(ConsumerConfig.GROUP_ID_CONFIG,groupId);
        gmKafkaConsumer.setProp(ConsumerConfig.MAX_POLL_RECORDS_CONFIG,"1000");

        if(isStartEarly){
            gmKafkaConsumer.getSource().setStartFromEarliest();
        }else if(startTime != null){
            SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            gmKafkaConsumer.getSource().setStartFromTimestamp(simpleDateFormat.parse(startTime).getTime());
        }

        SplitStream backend = env.addSource(gmKafkaConsumer.getSource()).map(new MapFunction<String, Tuple4<String,String,String,String>>() {
            //格式化数据
            @Override
            public Tuple4<String,String,String,String> map(String value) throws Exception {
                String contentString = value;
                JSONObject contentObject = JSON.parseObject(contentString);
                String module = contentObject.getString("MODULE");
                String time = contentObject.getString("TIME");
                JSONObject sysObject = contentObject.getJSONObject("SYS");
                String channel = null;
                if(!StringUtils.isObjectNull(sysObject)){
                    channel =sysObject.getString("channel");
                }
                return new Tuple4<String,String,String,String>(module,time,channel,contentString);
            }
        }).filter(new FilterFunction<Tuple4<String,String,String,String>>() {
            //对时间戳进行过滤
            @Override
            public boolean filter(Tuple4<String, String, String, String> value) throws Exception {
                long timestamp = 0L;
                try {
                    timestamp = dateTimeFormat.parseMillis(value._2());
                } catch (IllegalArgumentException e) {
                    try {
                        timestamp = dateTimeNoMillisFormat.parseMillis(value._2());
                    } catch (IllegalArgumentException e2) {
                        return false;
                    }
                }
                return true;
            }
        }).filter(new FilterFunction<Tuple4<String,String,String,String>>() {
            //对压测渠道进行过滤
            @Override
            public boolean filter(Tuple4<String, String, String, String> value) throws Exception {
                String module = value._1();
                String channel = value._2();
                if(module == null || "".equals(module)){
                    return false;
                }
                if(GM_INTERNAL_NON_PERSISTENT_CHANNEL.equals(channel)){
                    return false;
                }
                return true;
            }
        }).map(new MapFunction<Tuple4<String,String,String,String>,String>() {
            @Override
            public String map(Tuple4<String,String,String,String> value) throws Exception {
                return value._4();
            }
        }).filter(new FilterFunction<String>() {
            //过滤测试数据
            @Override
            public boolean filter(String value) throws Exception {
                if(JSON.isValid(value)){
                    JSONObject jsonObject = JSON.parseObject(value);
                    JSONObject sysObject = jsonObject.getJSONObject("SYS");
                    if(!StringUtils.isObjectNull(sysObject)){
                        String action = sysObject.getString("action");
                        if("/api/ok".equals(action)){
                            return false;
                        }else{
                            return true;
                        }
                    }else{
                        return false;
                    }
                }else {
                    return false;
                }
            }
        }).split(new OutputSelector<String>() {
            @Override
            public Iterable<String> select(String value) {
                JSONObject jsonObject = JSON.parseObject(value);
                String module = jsonObject.getString("MODULE");

                List<String> categroy = new ArrayList<>();
                categroy.add(module);
                return categroy;
            }
        });

        DataStream<BackendEtl> backendMaidianEtl = backend.select("backend").filter(new FilterFunction<String>() {
            @Override
            public boolean filter(String value) throws Exception {
                return JSON.isValid(value);
            }
        }).filter(new FilterFunction<String>() {
            //过滤app_name
            @Override
            public boolean filter(String value) throws Exception {
                JSONObject jsonObject = JSON.parseObject(value);
                JSONObject appObject = jsonObject.getJSONObject("APP");
                String antispam = "";
                if(!StringUtils.isObjectNull(appObject)){
                    antispam = StringUtils.changeNullTolength0(appObject.get("antispam"));
                }
                if("0".equals(antispam)){
                    if(appObject.get("name") == null){
                        return true;
                    }else{
                        return false;
                    }
                }else {
                    return false;
                }
            }
        }).map(new GainValueMap()).filter(new FilterFunction<BackendEtl>() {
            @Override
            public boolean filter(BackendEtl value) throws Exception {
                return (value.getCl_id() != null);
            }
        });

        DataStream<BackendEtl> result = backendMaidianEtl.filter(new FilterFunction<BackendEtl>() {
            //过滤设备id
            @Override
            public boolean filter(BackendEtl value) throws Exception {
                if((value.getCl_id() != null) && (!"".equals(value.getCl_id()) || !"".equals(value.getCl_idfv()))){
                    if(!"017746774461753".equals(value.getCl_id())){
                        return true;
                    }else{
                        return false;
                    }
                }else{
                    return false;
                }
            }
        }).keyBy(new KeySelector<BackendEtl, String>() {
                    @Override
                    public String getKey(BackendEtl backendEtl) throws Exception {
                        return backendEtl.getCl_id();
                    }});

        result.addSink(new KafkaSink(zxJdbcUrl,outJdbcUrl,outTable)).setParallelism(parallelism).uid("id-sink");
        env.execute("ml_device_backend");

    }
}
