//
//  HeadTranformIndexView.m
//  UnretekFace3D
//
//  Created by JasWorkSpace on 2018/10/12.
//  Copyright © 2018 JasWorkSpace. All rights reserved.
//

#import <Foundation/Foundation.h>
#include "HeadTranformIndexView.h"

/////////////////////////////////////////////////////////////////
@interface IndexView : UIView
@property(nonatomic,assign) BOOL selected;
@end
@implementation IndexView

- (instancetype)initWithFrame:(CGRect)frame
{
    self = [super initWithFrame:frame];
    if (self) {
        [self commitIndexViewInit];
    }
    return self;
}

- (void) commitIndexViewInit
{
    self.layer.cornerRadius = [self getRadiusFromFrame:self.frame];
    [self setSelected:NO andMustChangeColor:YES];
}

- (float) getRadiusFromFrame:(CGRect) frame
{
    if(self.frame.size.height > self.frame.size.width){
        return self.frame.size.width / 2;
    } else {
        return self.frame.size.height / 2;
    }
}

- (void)setSelected:(BOOL)selected
{
    [self setSelected:selected andMustChangeColor:NO];
}

- (void) setSelected:(BOOL)selected andMustChangeColor:(BOOL) must
{
    if(must || _selected != selected){
        float scale = (selected ? 0.8 : 0.6);
        [UIView transitionWithView:self duration:0.3 options:UIViewAnimationOptionTransitionCrossDissolve animations:^{
            self.transform = CGAffineTransformMakeScale(scale, scale);
            self.backgroundColor = (selected ? [UIColor greenColor] : [UIColor lightGrayColor]);
        } completion:nil];
            
    }
    _selected = selected;
}
@end
/////////////////////////////////////////////////////////////////
static const int FLAG_U = 0;//unkonw
static const int FLAG_H = 1;//h
static const int FLAG_V = 2;//v

@interface HeadTranformIndexView()
@property(nonatomic,assign) int number;
@property(nonatomic,assign) int count;
@property(nonatomic,assign) int orientation;
//////////////////////////////////
@property(nonatomic,assign) float indexViewSize;
@property(nonatomic,assign) float indexViewPadding;
@property(nonatomic,assign) float arcWidth;
@property(nonatomic,assign) float calculateIndexViewMaxSize;
/////////////////
@property(nonatomic,strong) NSMutableArray *nagativeIndexViews;
@property(nonatomic,strong) NSMutableArray *positiveIndexViews;
@end

@implementation HeadTranformIndexView

- (instancetype)initWithFrame:(CGRect)frame
{
    self = [super initWithFrame:frame];
    if (self) {
        _count = 0;
        [self commitHeadTranformIndexViewInit];
    }
    return self;
}

- (instancetype)initWithFrame:(CGRect)frame andCount:(int) count
{
    self = [super initWithFrame:frame];
    if (self) {
        if(count > 0){
            _count = count;
        }
        [self commitHeadTranformIndexViewInit];
    }
    return self;
}

- (void) commitHeadTranformIndexViewInit
{
    _arcMaxSize = 30.f;
    [self setUpUI];
}

- (void)setArcMaxSize:(float)arcMaxSize
{
    if(arcMaxSize > 0 && arcMaxSize<_indexViewSize && arcMaxSize != _arcMaxSize){
        _arcMaxSize = arcMaxSize;
        if(_arcMaxSize != _indexViewSize && _arcMaxSize < _calculateIndexViewMaxSize){
            [self rebuildUI];
        }
    }
}
- (int) getMaxNumber
{
    return _count;
}
///////////////////////////////////////////////////////
- (NSString *)description
{
    return [NSString stringWithFormat:@"{frame:%@, number:%d, arcWidth:%f, orientation:%d, \
            indexViewSize:%f, indexViewPadding:%f}",
            NSStringFromCGRect(self.frame),
            _number,
            _arcWidth,
            _orientation,
            _indexViewSize,
            _indexViewPadding];
}

-(NSMutableArray *)nagativeIndexViews
{
    if(!_nagativeIndexViews){
        _nagativeIndexViews = [NSMutableArray new];
    }
    return _nagativeIndexViews;
}

-(NSMutableArray *)positiveIndexViews
{
    if(!_positiveIndexViews){
        _positiveIndexViews = [NSMutableArray new];
    }
    return _positiveIndexViews;
}
//////////////////////////////////////////////////////
- (void) rebuildUI
{
    [self setUpUI];
    [self setUpNagativeIndexView];
    [self setUpPositiveIndexView];
}

- (void) setUpUI
{
    //索引数量
    _number = 2*_count + 1;
    //中间索引的圆环宽度
    _arcWidth = 3.0f;
    //索引的UIView大小和间隔
    if(self.frame.size.height > self.frame.size.width){
        _orientation = FLAG_V;
        [self calculateIndexViewSizeWithMin:self.frame.size.width andMax:self.frame.size.height];
    } else {
        _orientation = FLAG_H;
        [self calculateIndexViewSizeWithMin:self.frame.size.height andMax:self.frame.size.width];
    }
    [self buildUI];
}

- (void) calculateIndexViewSizeWithMin:(float) min andMax:(float) max
{
    //已横向为例进行计算
    float maxSize_V = (min - 2*_arcWidth);
    float maxSize_H = (max - 2*_arcWidth) / _number;
    if(maxSize_H > maxSize_V){
        _calculateIndexViewMaxSize = maxSize_V;
    } else {
        _calculateIndexViewMaxSize = maxSize_H;
    }
    _indexViewSize = (_calculateIndexViewMaxSize > _arcMaxSize ? _arcMaxSize : _calculateIndexViewMaxSize);
    _indexViewPadding = (max - 2*_arcWidth - _indexViewSize*_number) / (_number + 1.f);
}

- (void) buildUI
{
    [self removeAllChildView];
    // add Center
    [self addCenterView];
    if(_orientation == FLAG_H){
        [self addLeftView];
        [self addRightView];
    } else if(_orientation == FLAG_V){
        [self addTopView];
        [self addButtomView];
    }
}

- (void) removeAllChildView
{
    @synchronized(self){
        [_positiveIndexViews removeAllObjects];
        [_nagativeIndexViews removeAllObjects];
    }
    NSEnumerator * enumerateor =  [self.subviews objectEnumerator];
    UIView * child;
    while (child = [enumerateor nextObject]) {
        [child removeFromSuperview];
    }
}

- (void) addLeftView
{
    @synchronized (self) {
        float start_X = (self.frame.size.width/2 - (_indexViewSize/2 + _arcWidth + _indexViewPadding) - _indexViewSize);
        float start_Y = (self.frame.size.height - _indexViewSize)/2;
        for(int index=0; index<_count; index++){
            IndexView * indexview = [[IndexView alloc] initWithFrame:CGRectMake(start_X, start_Y, _indexViewSize, _indexViewSize)];
            [self.nagativeIndexViews insertObject:indexview atIndex: index];
            start_X -= (_indexViewPadding + _indexViewSize);
            [self addSubview:indexview];
        }
    }
}

- (void) addRightView
{
    @synchronized (self) {
        float start_X = (self.frame.size.width/2 + (_indexViewSize/2 + _arcWidth + _indexViewPadding));
        float start_Y = (self.frame.size.height - _indexViewSize)/2;
        for(int index=0; index<_count; index++){
            IndexView * indexview = [[IndexView alloc] initWithFrame:CGRectMake(start_X, start_Y, _indexViewSize, _indexViewSize)];
            [self.positiveIndexViews insertObject:indexview atIndex: index];
            start_X += (_indexViewPadding + _indexViewSize);
            [self addSubview:indexview];
        }
    }
}

- (void) addButtomView
{
    @synchronized (self) {
        float start_Y = (self.frame.size.height/2 + ( _indexViewSize/2 + _arcWidth + _indexViewPadding));
        float start_X = (self.frame.size.width - _indexViewSize)/2;
        for(int index=0; index<_count; index++){
            IndexView * indexview = [[IndexView alloc] initWithFrame:CGRectMake(start_X, start_Y, _indexViewSize, _indexViewSize)];
            [self.nagativeIndexViews insertObject:indexview atIndex: index];
            start_Y += (_indexViewPadding + _indexViewSize);
            //NSLog(@"%d, %@", (_count-index), indexview);
            ////// add to view
            [self addSubview:indexview];
        }
    }
}

- (void) addTopView
{
    @synchronized (self) {
        float start_Y = (self.frame.size.height/2 - ( _indexViewSize/2 + _arcWidth + _indexViewPadding) - _indexViewSize);
        float start_X = (self.frame.size.width - _indexViewSize)/2;
        for(int index=0; index<_count; index++){
            IndexView * indexview = [[IndexView alloc] initWithFrame:CGRectMake(start_X, start_Y, _indexViewSize, _indexViewSize)];
            [self.positiveIndexViews insertObject:indexview atIndex: index];
            start_Y -= (_indexViewPadding + _indexViewSize);
            //NSLog(@"%d, %@", (_count-index), indexview);
            ////// add to view
            [self addSubview:indexview];
        }
    }
}
//////////////////////////////////////////////////////////
- (void) addCenterView
{
    if(_orientation == FLAG_H || _orientation == FLAG_V){
        UIView * centerView = [self buildCenterUiView];
        [self addSubview:centerView];
    }
}

- (UIView *) buildCenterUiView
{
    CGSize size = self.frame.size;
    float start_X = (size.width/2 - (_indexViewSize/2 + _arcWidth));
    float start_Y = (size.height/2 - (_indexViewSize/2 + _arcWidth));
    UIImageView * imageView = [[UIImageView alloc] initWithFrame:CGRectMake(start_X, start_Y, _indexViewSize + _arcWidth*2, _indexViewSize + _arcWidth*2)];
    imageView.image = [self imageWithSize:(_indexViewSize + _arcWidth) andArc:_arcWidth];
    return imageView;
}

- (UIImage *)imageWithSize:(float)size andArc:(float) arc
{
    UIGraphicsBeginImageContextWithOptions(CGSizeMake(size, size), NO, 0);
    //2、画大圆
    UIBezierPath *path = [UIBezierPath bezierPathWithOvalInRect:CGRectMake(0, 0, size, size)];
    [[UIColor whiteColor] set];
    [path fill];
    //2、画小圆
    UIBezierPath *minpath = [UIBezierPath bezierPathWithOvalInRect:CGRectMake(arc, arc, size-2*arc, size-2*arc)];
    [[UIColor greenColor] set];
    [minpath fill];
    UIImage *clipImage = UIGraphicsGetImageFromCurrentImageContext();
    //3、关闭上下文
    UIGraphicsEndImageContext();
    return clipImage;
}

////////////////////////////////////////////////
- (void) reSetIndex
{
    self.nagativeNumber = 0;
    self.positiveNumber = 0;
}

- (void)setNagativeNumber:(int)nagativeNumber
{
    if(nagativeNumber!=_nagativeNumber && nagativeNumber >=0 && nagativeNumber <= _count){
        _nagativeNumber = nagativeNumber;
        [self setUpNagativeIndexView];
    }
}

- (void) setUpNagativeIndexView
{
    //NSLog(@"setUpNagativeIndexView count:%zi,_nagativeNumber:%d", self.nagativeIndexViews.count,_nagativeNumber);
    for(int index=0; index<self.nagativeIndexViews.count; index++){
        IndexView * indexview = [self.nagativeIndexViews objectAtIndex:index];
        if(indexview){
            indexview.selected = (index<_nagativeNumber);
        }
    }
}

- (void)setPositiveNumber:(int)positiveNumber
{
    if(positiveNumber!=_positiveNumber && positiveNumber >=0 && positiveNumber <= _count){
        _positiveNumber = positiveNumber;
        [self setUpPositiveIndexView];
    }
}

- (void) setUpPositiveIndexView
{
    //NSLog(@"setUpPositiveIndexView count:%zi,_positiveNumber:%d", self.positiveIndexViews.count,_positiveNumber);
    for(int index=0; index<self.positiveIndexViews.count; index++){
        IndexView * indexview = [self.positiveIndexViews objectAtIndex:index];
        if(indexview){
            indexview.selected = (index<_positiveNumber);
        }
    }
}

@end
