//
//  FixHorzontalLayoutButton.swift
//  Gengmei
//
//  Created by wangyang on 16/5/16.
//  Copyright © 2016年 更美互动信息科技有限公司. All rights reserved.
//

import UIKit
//import GMBaseSwift

/**
 *  @author licong, 15-09-27 17:09:12
 *
 *  Button同时存在图片和文字时候的排列样式
 *
 *  @since 5.2.0
 */
@objc public enum ImageTitleType: Int {
    case imageLeftTitleRight = 0
    case imageRightTitleLeft = 1
    case imageTopTitleBottom = 2
    case imageBottomTitleTop = 3

    func isHorizontalLayout() -> Bool {
        if self.rawValue < 2 {
            return true
        } else {
            return false
        }
    }
}

/// 注意：不能在tableView或者collectionView中使用
@objcMembers
public class AllLayoutButton: GMButton {

    public var type = ImageTitleType.imageRightTitleLeft
    public var space: CGFloat = 5.0
    public var margin = UIEdgeInsets.zero

    public convenience init(image: UIImage?, title: String?, titleColor: UIColor?, font: UIFont?) {
        self.init()
        setTitle(title, for: .normal)
        setImage(image, for: .normal)
        setTitleColor(titleColor, for: .normal)
        if font != nil {
            titleLabel?.font = font
        }
    }

    override public func layoutSubviews() {
        super.layoutSubviews()

        self.titleEdgeInsets = UIEdgeInsets.zero
        self.imageEdgeInsets = UIEdgeInsets.zero

        /* 以水平模式来讲解。把titleLabel、image和space看作一个整体square，这个整体的大小正好可以用 intrinsicContentSize 来表示。
         centerX是button的中心点
         
         |----------------------------------|
         |                                  |
         |    |--------------|  |-------|   |
         |    |   Label      | s| Image |   |
         |    |              |  |       |   |
         |    |--------------|  |-------|   |
         |                                  |
         |----------------------------------|
                           |centerX
              |       squareSize        |
         */
        let buttonWidth = bounds.size.width
        let buttonHeight = bounds.size.height

        let centerX = buttonWidth / 2.0
        let centerY = buttonHeight / 2.0
        let squareSize = intrinsicContentSize
        let squareLeft = floor(centerX - squareSize.width / 2)
        let squareRight = floor(centerX + squareSize.width / 2)

        let spuareTop = floor(centerY - squareSize.height / 2)
        let spuareBottom = floor(centerY + squareSize.height / 2)
        switch type {
        case .imageLeftTitleRight:
            imageView?.left = squareLeft + margin.left
            titleLabel?.right = squareRight - margin.right
        case .imageRightTitleLeft:
            titleLabel?.left = squareLeft + margin.left
            imageView?.right = squareRight - margin.right
        case .imageTopTitleBottom:
            imageView?.top = spuareTop + margin.top
            imageView?.centerX = centerX
            titleLabel?.bottom = spuareBottom - margin.bottom
            titleLabel?.width = squareSize.width
            titleLabel?.centerX = centerX
        case .imageBottomTitleTop:
            titleLabel?.top = spuareTop + margin.top
            titleLabel?.width = squareSize.width
            titleLabel?.centerX = centerX
            imageView?.bottom = spuareBottom - margin.bottom
            imageView?.centerX = centerX
        }
    }

    override public var intrinsicContentSize: CGSize {
        var size = super.intrinsicContentSize
        size.width += margin.left + margin.right
        size.height += margin.top + margin.bottom
        let labelHeight = titleLabel?.intrinsicContentSize.height
        let labelWidth = titleLabel?.intrinsicContentSize.width
        let imageViewHeight = imageView?.intrinsicContentSize.height
        let imageViewWidth = imageView?.intrinsicContentSize.width
        if type.isHorizontalLayout() {
            size.width += space
        } else {
            //竖直image和title竖直排列
            size.height = labelHeight! + imageViewHeight! + space
            if labelWidth != nil && imageViewWidth != nil && labelWidth! > imageViewWidth! {
                size.width = labelWidth!
            } else {
                size.width = imageViewWidth!
            }
        }
        return size
    }

    // 强制刷新，否则 layout会不正确
    override public func setTitle(_ title: String?, for state: UIControl.State) {
        super.setTitle(title, for: state)
        self.layoutIfNeeded()
    }

    // 强制刷新，否则 layout会不正确
    override public func setImage(_ image: UIImage?, for state: UIControl.State) {
        super.setImage(image, for: state)
        self.layoutIfNeeded()
    }
    
}
