//
//  Phobos.m
//  GengmeiDoctor
//
//  Created by Thierry on 16/1/26.
//  Copyright © 2016年 wanmeizhensuo. All rights reserved.
//

#import "Phobos.h"
#import <AdSupport/AdSupport.h>
#import <zlib.h>
#import "WMCacheService.h"

#ifdef DEBUG
#define phobosLog(...) NSLog(@"[Phobos] %@",__VA_ARGS__)
#else
#define phobosLog(...)
#endif

#define PhobosHaveOpenApp     @"PhobosHaveOpenApp"        //是否打开过APP
#define PhobosBeginTime       @"PhobosBeginTime"          //记录APP打开|从后台启动时的时间戳
#define PhobosEndTime         @"PhobosEndTime"            //记录APP退出|退到后台时的时间戳
#define PhobosCacheKey        @"PhobosCacheKey"
#define PhobosShardCount      50                          //收集数据分段发送的个数

static Phobos *sharedClient = nil;
static NSString *sdkVersion = @"110";

@interface Phobos ()

@property (strong, nonatomic) NSDateFormatter *dateFormatter;
@property (strong, nonatomic) NSString *appName;
@property (strong, nonatomic) NSString *channelId;
@property (strong, nonatomic) NSString *appVersion;
@property (assign, nonatomic) NSInteger userId;
@property (assign, nonatomic) BOOL logEnabled;

@end

@implementation Phobos

+ (Phobos *)clientWithAppName:(NSString *)appName channelId:(NSString *)channelId{
    return [[self alloc] initWithAppName:appName channelId:channelId];
}

+ (Phobos *)sharedClient{
    return sharedClient;
}

+ (void)setSharedClient:(Phobos *)client{
    sharedClient = client;
}

- (instancetype)initWithAppName:(NSString *)appName channelId:(NSString *)channelId{
    self = [super init];
    if (self) {
        _appName = appName;
        _channelId = channelId;
        _logEnabled = NO;
        _userId = 0;
        _appVersion = [self getAppVersion];
        [self setupNotification];
        [self handleEventAfterInit];
        
        phobosLog(@"starts to orbit");
    }
    return self;
}

- (void)setLogEnabled:(BOOL)value{
    _logEnabled = value;
}

- (void)setUserId:(NSInteger)userId{
    _userId = userId;
}

- (void)handleEventAfterInit{
    WMCacheService *cache = [WMCacheService sharedInstance];
    
    /** 每次打开APP埋点 **/
    [self track:@"device_opened" attributes:@{} sendNow:YES];
    
    /** 第一次打开APP埋点 **/
    if (![cache fetchObjectAtDiskWithkey:PhobosHaveOpenApp]) {
        [self track:@"device_activated" attributes:@{} sendNow:YES];
        [cache storeObjectAtDiskWithkey:PhobosHaveOpenApp object:PhobosHaveOpenApp];
    }
}


#pragma - mark notification handler
/**
 *  @brief 设置对APP的通知监听
 *
 *  @since 0.0.1
 */
- (void)setupNotification{
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(handleAppFinishLaunch:) name:UIApplicationDidFinishLaunchingNotification object:nil];
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(handleAppInForeground:) name:UIApplicationWillEnterForegroundNotification object:nil];
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(handleAppInBackgound:) name:UIApplicationDidEnterBackgroundNotification object:nil];
}

/**
 *  @brief 每次打开APP或返回前台，即Session开始的时候的处理
 *  @notification didFinishLaunch和willEnterForeground的时候都需要记录
 *
 *  @since 0.0.1
 */
- (void)handleSessionStart{
    WMCacheService *cache = [WMCacheService sharedInstance];
    [cache storeObjectAtDiskWithkey:PhobosBeginTime object:[self currentTime]];
}

/**
 *  @brief 应用打开时的处理
 *
 *  @param sender
 *
 *  @since 0.0.1
 */
- (void)handleAppFinishLaunch:(id)sender{
    phobosLog(@"handleAppFinishLaunch");
    [self handleSessionStart];
}

/**
 *  @brief 应用进入前台的处理
 *
 *  @param sender
 *
 *  @since 0.0.1
 */
- (void)handleAppInForeground:(id)sender{
    phobosLog(@"handleAppInForeground");
    [self handleSessionStart];
    [self fetchDataAndSend];
}

/**
 *  @brief 应用进入后台的处理
 *
 *  @param sender
 *
 *  @since 0.0.1
 */
- (void)handleAppInBackgound:(id)sender{
    phobosLog(@"handleAppInBackgound");
    [self handleSessionOver];
    [self fetchDataAndSend];
}

/**
 *  @brief 会话结束时候的处理
 *
 *  @since 0.0.1
 */
- (void)handleSessionOver{
    WMCacheService *cache = [WMCacheService sharedInstance];
    //进入后台的同时，把记录时间同步到服务端，把已使用时间清空
    double beginTime = [[cache fetchObjectAtDiskWithkey:PhobosBeginTime] doubleValue];
    if (beginTime == 0) {
        return;
    }
    NSDate *date = [NSDate date];
    double endTime = [date timeIntervalSince1970];
    NSString *usedTime = [NSString stringWithFormat:@"%f",endTime - beginTime];
    NSMutableDictionary *dict = [NSMutableDictionary dictionaryWithObjectsAndKeys:usedTime,@"duration",nil];
    [self track:@"on_app_session_over" attributes:dict];
    [cache removeObjectAtDiskWithkey:PhobosBeginTime];
}


#pragma - mark track event handler
- (void)track:(NSString *)eventId{
    [self track:eventId attributes:@{}];
}

- (void)track:(NSString *)eventId attributes:(NSDictionary *)attributes{
    NSArray *array = [[WMCacheService sharedInstance] fetchObjectAtDiskWithkey:PhobosCacheKey];
    //超过一定数量的话，统一发送一次
    if (array.count > PhobosShardCount) {
        [self sendArray:array];
    }
    [self track:eventId attributes:attributes sendNow:NO];
}

- (void)track:(NSString *)eventId attributes:(NSDictionary *)attributes sendNow:(BOOL)sendNow{
    NSDictionary *dict = [self prepareDictionaryForEvent:eventId attributes:attributes];
    if (sendNow) {
        NSArray *array = @[dict];
        [self sendArray:array];
    }else{
        [self save:dict];
    }
}

- (void)trackJsEvent:(NSString *)jsonString{
    @try {
        NSData *data = [jsonString dataUsingEncoding:NSUnicodeStringEncoding];
        NSMutableDictionary *dict = [NSJSONSerialization JSONObjectWithData:data options:NSJSONReadingAllowFragments error:nil];
        
        id pa = dict[@"params"];
        NSDictionary *json;
        if ([pa isKindOfClass:[NSString class]]) {
            NSError *jsonError;
            NSData *objectData = [pa dataUsingEncoding:NSUTF8StringEncoding];
            json = [NSJSONSerialization JSONObjectWithData:objectData
                                                   options:NSJSONReadingMutableContainers
                                                     error:&jsonError];
            [self track:dict[@"type"] attributes:json];
        }else{
            [self track:dict[@"type"]];
        }
    }
    @catch (NSException *exception) {
        phobosLog(exception);
    }
}


/**
 *  @brief 将埋点时间封装成词典数据
 *
 *  @return
 *
 *  @since 0.0.1
 */
- (NSDictionary *)prepareDictionaryForEvent:(NSString *)eventId attributes:(NSDictionary *)attributes{
    NSMutableDictionary *dict = [[NSMutableDictionary alloc] init];
    @try {
        NSString *currentTime = [self currentTime];
        NSMutableDictionary *deviceParams = [NSMutableDictionary dictionaryWithObjectsAndKeys:
                                             [[[ASIdentifierManager sharedManager] advertisingIdentifier] UUIDString],@"device_id",
                                             @"ios",@"device_type",nil];
        NSMutableDictionary *appParams   = [NSMutableDictionary dictionaryWithObjectsAndKeys:
                                            self.appName, @"name",
                                            self.appVersion, @"version",
                                            self.channelId,@"channel", nil];
        [dict setObject:eventId forKey:@"type"];
        [dict setObject:appParams forKey:@"app"];
        [dict setObject:sdkVersion forKey:@"version"];
        [dict setObject:deviceParams forKey:@"device"];
        [dict setObject:@(_userId) forKey:@"user_id"];
        [dict setObject:currentTime forKey:@"create_at"];
        [dict setObject:attributes forKey:@"params"];
    }
    @catch (NSException *exception) {
        phobosLog(exception);
    }
    return dict;
}

/**
 *  @brief 保存数据到缓存层
 *
 *  @param data 数据
 *
 *  @since 0.0.1
 */
- (void)save:(NSDictionary *)data
{
    if (_logEnabled) {
        phobosLog([NSString stringWithFormat:@"save dictionary: %@",data]);
    }
    WMCacheService *cache = [WMCacheService sharedInstance];
    NSMutableArray *dataArray = [cache fetchObjectAtDiskWithkey:PhobosCacheKey];
    if (dataArray) {
        [dataArray addObject:data];
    }else{
        dataArray = [NSMutableArray arrayWithObject:data];
    }
    [cache storeObjectAtDiskWithkey:PhobosCacheKey object:dataArray];
}

/**
 *  @brief 从缓存中获取数据，并发送
 *
 *  @since 0.0.1
 */
- (void)fetchDataAndSend{
    NSArray *paramsArray = [[WMCacheService sharedInstance] fetchObjectAtDiskWithkey:PhobosCacheKey];
    if (paramsArray.count>0) {
        [self sendArray:paramsArray];
    }
}


/**
 *  @brief 发送数组
 *
 *  @param array
 *
 *  @since 0.0.1
 */
- (void)sendArray:(NSArray *)array {
    if (_logEnabled) {
        phobosLog([NSString stringWithFormat:@"array prepare to fly --✈: %@",array]);
    }
    @try {
        NSData *JSON = [self encodeJSON:array];
        NSData *compressedData = [self compressData:JSON];
        [self sendData:compressedData];
    }
    @catch (NSException *exception) {
        phobosLog(exception);
    }
}

/**
 *  @brief 上传数据
 *
 *  @param data
 *
 *  @since 0.0.1
 */
- (void)sendData:(NSData *)data {
#ifdef DEBUG
    NSString *url = @"http://log.test.gengmei.cc/log/collect";
#else
    NSString *url = @"http://log.gengmei.cc/log/collect";
#endif
    @try {
        NSMutableURLRequest *request = [[NSMutableURLRequest alloc] initWithURL:[NSURL URLWithString:url]];
        [request setValue:@"gzip" forHTTPHeaderField:@"Content-Encoding"];
        request.HTTPBody = data;
        request.HTTPMethod = @"POST";
        [NSURLConnection sendAsynchronousRequest:request queue:[NSOperationQueue mainQueue] completionHandler:^(NSURLResponse * _Nullable response, NSData * _Nullable data, NSError * _Nullable connectionError) {
            NSHTTPURLResponse *res = (NSHTTPURLResponse *)response;
            NSInteger code = res.statusCode;
            if (code == 200) {
                phobosLog(@"✈ ---------- ✈ data arrived Mars");
                [[WMCacheService sharedInstance] removeObjectAtDiskWithkey:PhobosCacheKey];
            }
        }];
    }
    @catch (NSException *exception) {
        phobosLog(exception);
    }
}


/**
 *  @brief 压缩待上传的数据。经过测试，50条数据压缩大概6毫秒，所以不要放在异步线程中处理
 *
 *  @param originData 压缩前的数据
 *
 *  @return 压缩后的数据
 *
 *  @since 0.0.1
 */
- (NSData *)compressData:(NSData *)originData{
    if (!originData || [originData length] == 0)
    {
        if (_logEnabled) {
            phobosLog(@"Error: Can't compress an empty or null NSData object.");
        }
        return nil;
    }
    
    z_stream zlibStreamStruct;
    zlibStreamStruct.zalloc    = Z_NULL; // Set zalloc, zfree, and opaque to Z_NULL so
    zlibStreamStruct.zfree     = Z_NULL; // that when we call deflateInit2 they will be
    zlibStreamStruct.opaque    = Z_NULL; // updated to use default allocation functions.
    zlibStreamStruct.total_out = 0; // Total number of output bytes produced so far
    zlibStreamStruct.next_in   = (Bytef*)[originData bytes]; // Pointer to input bytes
    zlibStreamStruct.avail_in  = (uInt)[originData length]; // Number of input bytes left to process
    
    int initError = deflateInit2(&zlibStreamStruct, Z_DEFAULT_COMPRESSION, Z_DEFLATED, (15+16), 8, Z_DEFAULT_STRATEGY);
    if (initError != Z_OK)
    {
        NSString *errorMsg = nil;
        if (_logEnabled) {
            switch (initError)
            {
                case Z_STREAM_ERROR:
                    errorMsg = @"Invalid parameter passed in to function.";
                    break;
                case Z_MEM_ERROR:
                    errorMsg = @"Insufficient memory.";
                    break;
                case Z_VERSION_ERROR:
                    errorMsg = @"The version of zlib.h and the version of the library linked do not match.";
                    break;
                default:
                    errorMsg = @"Unknown error code.";
                    break;
            }
        }
        return nil;
    }
    NSMutableData *compressedData = [NSMutableData dataWithLength:[originData length] * 1.01 + 12];
    NSInteger deflateStatus;
    do
    {
        zlibStreamStruct.next_out = [compressedData mutableBytes] + zlibStreamStruct.total_out;
        zlibStreamStruct.avail_out = (uInt)([compressedData length] - zlibStreamStruct.total_out);
        deflateStatus = deflate(&zlibStreamStruct, Z_FINISH);
        
    } while ( deflateStatus == Z_OK );
    
    // Check for zlib error and convert code to usable error message if appropriate
    if (deflateStatus != Z_STREAM_END)
    {
        NSString *errorMsg = nil;
        if (_logEnabled) {
            switch (deflateStatus)
            {
                case Z_ERRNO:
                    errorMsg = @"Error occured while reading file.";
                    break;
                case Z_STREAM_ERROR:
                    errorMsg = @"The stream state was inconsistent (e.g., next_in or next_out was NULL).";
                    break;
                case Z_DATA_ERROR:
                    errorMsg = @"The deflate data was invalid or incomplete.";
                    break;
                case Z_MEM_ERROR:
                    errorMsg = @"Memory could not be allocated for processing.";
                    break;
                case Z_BUF_ERROR:
                    errorMsg = @"Ran out of output buffer for writing compressed bytes.";
                    break;
                case Z_VERSION_ERROR:
                    errorMsg = @"The version of zlib.h and the version of the library linked do not match.";
                    break;
                default:
                    errorMsg = @"Unknown error code.";
                    break;
            }
        }
        deflateEnd(&zlibStreamStruct);
        return nil;
    }
    deflateEnd(&zlibStreamStruct);
    [compressedData setLength: zlibStreamStruct.total_out];
    return compressedData;
}

#pragma mark - helpers
/**
 *  @brief 将对象转成JSON格式数据
 *
 *  @param obj
 *
 *  @return
 *
 *  @since 0.0.1
 */
- (NSData *)encodeJSON:(id)obj {
    NSData *data = [NSJSONSerialization dataWithJSONObject:obj options:0 error:nil];
    return data;
}


/**
 *  @brief 获取当前时间的毫秒数
 *
 *  @return
 *
 *  @since 0.0.1
 */
- (NSString *)currentTime {
    NSDate *date = [NSDate date];
    NSTimeInterval interval = [date timeIntervalSince1970];
    NSString *timeIntervalStr = [NSString stringWithFormat:@"%.0f",interval];
    return timeIntervalStr;
}

/**
 *  @brief 获取当前APP的版本号
 *
 *  @return
 *
 *  @since 0.0.1
 */
- (NSString *)getAppVersion{
    NSString *buildVersion = [[[NSBundle mainBundle] infoDictionary] objectForKey:@"CFBundleShortVersionString"];
    if (buildVersion) {
        return buildVersion;
    }else{
        return @"";
    }
}

- (void)dealloc{
    if (self) {
        [[NSNotificationCenter defaultCenter] removeObserver:self];
    }
}


@end
