//
//  GMUILabel.m
//  Gengmei
//
//  Created by Thierry on 12/26/14.
//  Copyright (c) 2014 Wanmeichuangyi. All rights reserved.
//

#import <GMKit/GMLabel.h>
#import "GMFont.h"
#import "GMConstant.h"
@implementation GMLabel

- (instancetype)initWithFrame:(CGRect)frame
{
    self = [super initWithFrame:frame];
    if (self) {
        [self setup];
    }
    return self;
}

- (void)awakeFromNib{
    [super awakeFromNib];
    [self setup];
}

- (void)setup{
    _verticalAlignment = GMLabelVerticalAlignmentMiddle;
}

+ (GMLabel *)labelWithTextColor:(UIColor *)color fontSize:(CGFloat)fontSize{
    GMLabel *lable = [GMLabel new];
    lable.textColor = color;
    lable.font = GMFont(fontSize);
    return lable;
}

+ (GMLabel *)labelWithTextAlignment:(NSTextAlignment)textAlignment backgroundColor:(UIColor*)backgroundColor textColor:(UIColor*)textColor fontSize:(CGFloat )fontSize{
    GMLabel *lable = [GMLabel new];
    lable.textAlignment = textAlignment;
    lable.textColor = textColor;
    lable.backgroundColor = backgroundColor;
    lable.font = GMFont(fontSize);
    
    return lable;
}

- (void)setTextAlignment:(NSTextAlignment)textAlignment backgroundColor:(UIColor*)backgroundColor textColor:(UIColor*)textColor  fontSize:(CGFloat )fontSize{
    
    self.textAlignment = textAlignment;
    self.textColor = textColor;
    self.backgroundColor = backgroundColor;
    self.font = GMFont(fontSize);
}

- (void)setVerticalAlignment:(GMLabelVerticalAlignment)verticalAlignment {
    _verticalAlignment = verticalAlignment;
    [self setNeedsDisplay];
}


/**
 *  @author wangyang, 16-01-08 10:01:26
 *
 *  @brief  这个方法在每次设置新的text值时会调用一次，并且只在最后调用一次。也就是说此时已经有了正确的size
 *  @note 不经重写的 UILabel
 先调用 intrinsicContentSize
 再调用 textRectForBounds:limitedToNumberOfLines
 最后调用drawTextInRect
 *  @param requestedRect 等于self.bounds
 *  @since 5.9.0
 */
-(void)drawTextInRect:(CGRect)requestedRect {
    
    // 先去除paddingEdge，剩下的rect就是用来放 string 的 rect。但注意：未必是正好能容纳下string的大小的rect
    CGRect contentRect = UIEdgeInsetsInsetRect(requestedRect, self.paddingEdge);
    
    // 在contentRect内，string所需要的text rect
    CGRect textRect = [self textRectForBounds:contentRect limitedToNumberOfLines:self.numberOfLines];
    // 将坐标信息清0，由我们自己控制
    textRect.origin.x = 0;
    textRect.origin.y = 0;
    
    
    switch (self.verticalAlignment) {
        case GMLabelVerticalAlignmentTop: {
            textRect.origin.y = 0;
            break;
        }
        case GMLabelVerticalAlignmentBottom: {
            textRect.origin.y = requestedRect.size.height - textRect.size.height;
            break;
        }
        case GMLabelVerticalAlignmentMiddle:
        default: {
            textRect.origin.y = (requestedRect.size.height - textRect.size.height)/2;
            break;
        }
    }
    
    switch (self.textAlignment) {
            
        case NSTextAlignmentRight: {
            textRect.origin.x = requestedRect.size.width - textRect.size.width;
            break;
        }
        case NSTextAlignmentCenter: {
            textRect.origin.x = (requestedRect.size.width - textRect.size.width)/2;
            break;
        }
        case NSTextAlignmentNatural:
        case NSTextAlignmentLeft:
        default: {
            textRect.origin.x = 0;
            break;
        }
    }
    
    [super drawTextInRect:textRect];
}

// 因为有paddingEdge，需要重写 intrinsicContentSize 方法。如果在约束里指定了size，该方法会失效
- (CGSize)intrinsicContentSize {
    CGSize size = [super intrinsicContentSize];
    CGSize newSize = CGSizeMake(ceilf(size.width + self.paddingEdge.left + self.paddingEdge.right), ceilf(size.height + self.paddingEdge.top + self.paddingEdge.bottom));
    
    return newSize;
}

@end
