//
//  GMSQliteManager.h
//  GMSQlite
//
//  Created by 卢悦明 on 2020/6/10.
//  Copyright © 2020 卢悦明. All rights reserved.
//

#import <Foundation/Foundation.h>
#import <MJExtension/MJExtension.h>

// 数据库中常见的几种类型
#define SQL_TEXT     @"TEXT" //文本
#define SQL_INTEGER  @"INTEGER" //int long integer ...
#define SQL_REAL     @"REAL" //浮点
#define SQL_BLOB     @"BLOB" //data

@interface GMSQliteManager : NSObject


#pragma mark - 创建表
/// 创建表
/// @param tableName 名称
/// @param modelClass 类对象
- (BOOL)gm_createTable:(NSString *)tableName model:(Class)modelClass;

/// 创建表
/// @param tableName 名称
/// @param dict key 为键， value为存储类型
- (BOOL)gm_createTable:(NSString *)tableName dict:(NSDictionary *)dict;

#pragma mark - 插入操作
///字典插入
- (BOOL)gm_insertTable:(NSString *)tableName dict:(NSDictionary *)dict;
/// 模型插入
- (BOOL)gm_insertTable:(NSString *)tableName model:(NSObject *)model;
/**
 批量插入或更改
 @param dicOrModelArray 要insert/update数据的数组,也可以将model和dictionary混合装入array
 @return 返回的数组存储未插入成功的下标,数组中元素类型为NSNumber
 */
- (NSArray *)gm_insertTable:(NSString *)tableName dicOrModelArray:(NSArray *)dicOrModelArray;

#pragma mark - 删除操作
/// 删除表中某些元素，
/// @param tableName 表的名称
/// @param format 条件语句, 如:@"where name = '小李'"
- (BOOL)gm_deleteTable:(NSString *)tableName whereFormat:(NSString *)format;
///删除表
- (BOOL)gm_deleteTable:(NSString *)tableName;
///清空表
- (BOOL)gm_deleteAllDataFromTable:(NSString *)tableName;

#pragma mark - 修改操作
/// 根据条件更改表中数据
/// @param tableName 名称
/// @param parameters 要更改的数据,可以是model或dictionary(格式:@{@"name":@"张三"})
/// @param format 条件语句, 如:@"where name = '小李'"
- (BOOL)gm_updateTable:(NSString *)tableName
            dicOrModel:(id)parameters
           whereFormat:(NSString *)format;

#pragma mark - 查找操作
/**
 查找: 根据条件查找表中数据
 @param tableName 表的名称
 @param filterArray 过滤字段
 @param cls [Persen class]
 @param format 条件语句, 如:@"where name = '小李'",
 @return 将结果存入array, @[@{@"age":@"", @"name":@""}]
 */
- (NSArray *)gm_lookupTable:(NSString *)tableName
                filterArray:(NSArray *)filterArray
                objectClass:(Class)cls
                whereFormat:(NSString *)format;

- (NSArray *)gm_lookupTable:(NSString *)tableName
                   keyArray:(NSArray *)keyArray
                whereFormat:(NSString *)format;

#pragma mark - 辅助接口
///是否存在表
- (BOOL)gm_isExistTable:(NSString *)tableName;
/// 表中共有多少条数据
/// @param tableName 表的名称
/// @param format 筛选条件（where name = '小李'）, 如果没有条件可以传nil
- (int)gm_tableItemCount:(NSString *)tableName whereFormat:(NSString *)format;
/// 返回表中的字段名
- (NSArray *)gm_columnNameArray:(NSString *)tableName;
/// `关闭数据库
- (void)close;
/// 打开数据库 (每次shareDatabase系列操作时已经open,当调用close后若进行db操作需重新open或调用shareDatabase)
- (void)open;

/**
 增加新字段, 在建表后还想新增字段,可以在原建表model或新model中新增对应属性,然后传入即可新增该字段,该操作已在事务中执行
 
 @param tableName 表的名称
 @param parameters dictionary格式为@{@"newname":@"TEXT"}
 @return 是否成功
 */
- (BOOL)gm_alterTable:(NSString *)tableName keyDict:(NSDictionary *)parameters;

// =============================   线程安全操作    ===============================
#pragma mark - 线程安全的处理
/**
 将操作语句放入block中即可保证线程安全, 如:
 
 Person *p = [[Person alloc] init];
 p.name = @"小李";
 [db gm_inDatabase:^{
    [db gm_insertTable:@"users" dicOrModel:p];
 }];
 */
- (void)gm_inDatabase:(void (^)(void))block;

/**
 事务: 将操作语句放入block中可执行回滚操作(*rollback = YES;)
 Person *p = [[Person alloc] init];
 p.name = @"小李";
 for (int i=0,i < 1000,i++) {
 [db gm_inTransaction:^(BOOL *rollback) {
 BOOL flag = [db gm_insertTable:@"users" dicOrModel:p];
 if (!flag) {
 *rollback = YES; //只要有一次不成功,则进行回滚操作
 return;
 }
 }];
 }
 */
- (void)gm_inTransaction:(void(^)(BOOL *rollback))block;
@end

