//
//  GMSQliteManager.m
//  GMSQlite
//
//  Created by 卢悦明 on 2020/6/10.
//  Copyright © 2020 卢悦明. All rights reserved.
//

#import "GMSQliteManager.h"
#import <objc/runtime.h>
#import "FMDB.h"


@interface GMSQliteManager()

/// 路径
@property (nonatomic, strong)NSString *dbPath;
/// 队列
@property (nonatomic, strong)FMDatabaseQueue *dbQueue;
/// 数据库对象
@property (nonatomic, strong)FMDatabase *db;
@end

@implementation GMSQliteManager

- (instancetype)init
{
    self = [super init];
    if (self) {
        NSString * path = [[NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES) lastObject] stringByAppendingPathComponent:@"GMFMDB.sqlite"];
        NSLog(@"path------%@", path);
        FMDatabase *fmdb = [FMDatabase databaseWithPath:path];
        self.db = fmdb;
        self.dbPath = path;
        FMDatabaseQueue *queue = [FMDatabaseQueue databaseQueueWithPath:path];
        self.dbQueue = queue;
        [self.db open];
    }
    if (!self.db.isOpen) {
        [self.db open];
    }
    return self;
}


/// 创建表
- (BOOL)gm_createTable:(NSString *)tableName model:(Class)modelClass {
    NSDictionary *dict = [self modelToDictionary:modelClass excludePropertyName:nil];
    return [self gm_createTable:tableName dict:dict];
}

- (BOOL)gm_createTable:(NSString *)tableName dict:(NSDictionary *)dict {
 
    NSMutableString *fieldStr = [[NSMutableString alloc] initWithFormat:@"create table if not exists %@ (", tableName];
    
    for (NSString *key in dict.allKeys) {
        [fieldStr appendFormat:@"%@ %@, ", key, dict[key]];
    }
    if (dict.allKeys.count > 0) {
        [fieldStr replaceCharactersInRange:NSMakeRange(fieldStr.length - 2, 1) withString:@")"];
    } else {
        [fieldStr appendFormat:@")"];
    }
    
    BOOL creatFlag = [_db executeUpdate:fieldStr];
    return creatFlag;
}

#pragma mark - 插入操作

/// 模型插入
- (BOOL)gm_insertTable:(NSString *)tableName model:(NSObject *)model {
    //模型转字典
    
    NSDictionary *dict = model.mj_keyValues;;
    bool status =  [self gm_insertTable:tableName dict:dict];
    return status;
}

///字典插入
- (BOOL)gm_insertTable:(NSString *)tableName dict:(NSDictionary *)dict {
    //获取表中所有的字段
    NSArray *columnArr = [self getColumnArr:tableName db:_db];
    BOOL flag;
    NSMutableString *finalStr = [[NSMutableString alloc] initWithFormat:@"INSERT INTO %@ (", tableName];
    NSMutableString *tempStr = [NSMutableString stringWithCapacity:0];
    NSMutableArray *argumentsArr = [NSMutableArray arrayWithCapacity:0];
    for (NSString *key in dict) {
        if (![columnArr containsObject:key] ) {
            continue;
        }
        [finalStr appendFormat:@"%@,", key];
        [tempStr appendString:@"?,"];
        [argumentsArr addObject:dict[key]];
    }
    [finalStr deleteCharactersInRange:NSMakeRange(finalStr.length-1, 1)];
    if (tempStr.length){
        [tempStr deleteCharactersInRange:NSMakeRange(tempStr.length-1, 1)];
    }
    [finalStr appendFormat:@") values (%@)", tempStr];
    flag = [_db executeUpdate:finalStr withArgumentsInArray:argumentsArr];
    return flag;
}

///批量插入
- (NSArray *)gm_insertTable:(NSString *)tableName dicOrModelArray:(NSArray *)dicOrModelArray {
    NSMutableArray *array = [NSMutableArray array];
    for (int i = 0; i< dicOrModelArray.count; i++) {
        id sub = dicOrModelArray[i];
        BOOL status = NO;
        if ([sub isKindOfClass:[NSDictionary class]]) {
            status = [self gm_insertTable:tableName dict:(NSDictionary *)sub];
            if (status) {
                [array addObject:@(i)];
            }
        } else if ([sub isKindOfClass:[NSObject class]]) {
           status = [self gm_insertTable:tableName model:(NSObject *)sub];
            if (status) {
                [array addObject:@(i)];
            }
        }
    }
    return array;
}
#pragma mark - 删除操作
//按照条件删除
- (BOOL)gm_deleteTable:(NSString *)tableName whereFormat:(NSString *)format {
    if (![format containsString:@"where"] && ![format containsString:@"WHERE"]) {
        NSLog(@"sqlit语句不正确");
        return NO;
    }
    NSMutableString *finalStr = [[NSMutableString alloc] initWithFormat:@"delete from %@  %@", tableName,format];
    BOOL flag = [_db executeUpdate:finalStr];
    return flag;
}

//删除表格
- (BOOL)gm_deleteTable:(NSString *)tableName {
    NSString *sqlstr = [NSString stringWithFormat:@"DROP TABLE %@", tableName];
    if (![_db executeUpdate:sqlstr])
    {
        return NO;
    }
    return YES;
}

//清空表格
- (BOOL)gm_deleteAllDataFromTable:(NSString *)tableName {
    NSString *sqlstr = [NSString stringWithFormat:@"DELETE FROM %@", tableName];
    if (![_db executeUpdate:sqlstr])
    {
        return NO;
    }
    return YES;
}

#pragma mark - 修改操作
/// 根据条件更改表中数据
/// @param tableName 名称
/// @param parameters 要更改的数据,可以是model或dictionary(格式:@{@"name":@"张三"})
/// @param format 条件语句, 如:@"where name = '小李'"
- (BOOL)gm_updateTable:(NSString *)tableName
            dicOrModel:(id)parameters
           whereFormat:(NSString *)format {
    NSString *where = format;
    BOOL flag;
    NSDictionary *dict;
    NSArray *clomnArr = [self getColumnArr:tableName db:_db];
    if ([parameters isKindOfClass:[NSDictionary class]]) {
        dict = parameters;
    } else if([parameters isKindOfClass:[NSObject class]]) {
        NSObject *object =(NSObject *)parameters;
        dict = object.mj_keyValues;
    } else {
        return NO;
    }
    
    NSMutableString *finalStr = [[NSMutableString alloc] initWithFormat:@"update %@ set ", tableName];
    NSMutableArray *argumentsArr = [NSMutableArray arrayWithCapacity:0];
    
    for (NSString *key in dict) {
        if (![clomnArr containsObject:key]) {
            continue;
        }
        [finalStr appendFormat:@"%@ = %@,", key, @"?"];
        [argumentsArr addObject:dict[key]];
    }
    
    [finalStr deleteCharactersInRange:NSMakeRange(finalStr.length-1, 1)];
    if (where.length){
        [finalStr appendFormat:@" %@", where];
    }
    flag =  [_db executeUpdate:finalStr withArgumentsInArray:argumentsArr];
    
    return flag;
}


#pragma mark - 查找操作
- (NSArray *)gm_lookupTable:(NSString *)tableName
                filterArray:(NSArray *)filterArray
                objectClass:(Class)cls
                whereFormat:(NSString *)format {
    NSDictionary *modlDict = [self modelToDictionary:cls excludePropertyName:filterArray];
    NSArray *array = [self gm_lookupTable:tableName keyArray:modlDict.allKeys whereFormat:format];
    NSMutableArray *modelArray = [cls mj_objectArrayWithKeyValuesArray:array];
    return modelArray;
}

- (NSArray *)gm_lookupTable:(NSString *)tableName keyArray:(NSArray *)keyArray whereFormat:(NSString *)format {
    NSString *where = format;
    NSMutableString *finalStr = [[NSMutableString alloc] initWithFormat:@"select * from %@ %@", tableName, where ? where : @""];
    FMResultSet *set = [_db executeQuery:finalStr];
    NSMutableArray *resultMArr = [self dealArray:keyArray FMResultSet:set];
    return resultMArr;
}

- (NSMutableArray *)dealArray:(NSArray *)keyArray FMResultSet:(FMResultSet *)set{
    NSMutableArray *resultMArr = [NSMutableArray array];
    while ([set next]) {
        
        NSMutableDictionary *dict = [NSMutableDictionary dictionary];
        for (int i = 0; i< keyArray.count; i++) {
            if ([set stringForColumn:keyArray[i]]) {
                [dict setObject:[set stringForColumn:keyArray[i]] forKey:keyArray[i]];
            }
        }
        [resultMArr addObject:dict];
    }
    return  resultMArr;
}

#pragma mark - 辅助接口
///是否存在表
- (BOOL)gm_isExistTable:(NSString *)tableName {
    FMResultSet *set = [_db executeQuery:@"SELECT count(*) as 'count' FROM sqlite_master WHERE type ='table' and name = ?", tableName];
    while ([set next])
    {
        NSInteger count = [set intForColumn:@"count"];
        if (count == 0) {
            return NO;
        } else {
            return YES;
        }
    }
    return NO;
}
/// `表中共有多少条数据
- (int)gm_tableItemCount:(NSString *)tableName whereFormat:(NSString *)format {
    
    NSString *sqlstr = [NSString stringWithFormat:@"SELECT count(*) as 'count' FROM %@", tableName];
    if (format) {
        sqlstr = [sqlstr stringByAppendingFormat:@" %@", format];
    }
    
    FMResultSet *set = [_db executeQuery:sqlstr];
    while ([set next])
    {
        return [set intForColumn:@"count"];
    }
    return 0;
}
/// 返回表中的字段名
- (NSArray *)gm_columnNameArray:(NSString *)tableName {
    return [self getColumnArr:tableName db:_db];
}
/// `关闭数据库
- (void)close {
    [_db close];
}
/// 打开数据库 (每次shareDatabase系列操作时已经open,当调用close后若进行db操作需重新open或调用shareDatabase)
- (void)open {
    [_db open];
}

#pragma mark - 在表中增加字段
- (BOOL)gm_alterTable:(NSString *)tableName keyDict:(NSDictionary *)parameters {
    __block BOOL flag;
    [self gm_inTransaction:^(BOOL *rollback) {
        for (NSString *key in parameters) {
            NSString *squlStr = [NSString stringWithFormat:@"ALTER TABLE %@ ADD COLUMN %@ %@", tableName, key, parameters[key]];
            
            flag = [self->_db executeUpdate:squlStr];
            NSLog(@"**********%d", flag);
            if (!flag) {
                *rollback = YES;
                return;
            }
        }
    }];
    return flag;
}

#pragma mark - 线程安全的
- (void)gm_inDatabase:(void(^)(void))block
{
    [_dbQueue inDatabase:^(FMDatabase *db) {
        self.db = db;
        block();
    }];
}

- (void)gm_inTransaction:(void(^)(BOOL *rollback))block
{
    [_dbQueue inTransaction:^(FMDatabase *db, BOOL *rollback) {
        self.db = db;
        block(rollback);
    }];
}

#pragma mark - 通过类对象获取对象的Property的字典 key 是属性名称， value是属性的类型
- (NSDictionary *)modelToDictionary:(Class)cls excludePropertyName:(NSArray *)nameArr
{
    if (cls == [NSObject class]) {
        return [NSMutableDictionary dictionary];
    }
    NSMutableDictionary *mDic = [[NSMutableDictionary alloc] init];
    unsigned int outCount;
    objc_property_t *properties = class_copyPropertyList(cls, &outCount);
    for (int i = 0; i < outCount; i++) {
        NSString *name = [NSString stringWithCString:property_getName(properties[i]) encoding:NSUTF8StringEncoding];
        if ([nameArr containsObject:name]) continue;
        NSString *type = [NSString stringWithCString:property_getAttributes(properties[i]) encoding:NSUTF8StringEncoding];
        id value = [self propertTypeConvert:type];
        if (value) {
            [mDic setObject:value forKey:name];
        }
    }
    free(properties);
    NSDictionary *subDict = [self modelToDictionary:[cls superclass] excludePropertyName:nameArr];
    [mDic addEntriesFromDictionary:subDict];
    return mDic;
}

#pragma mark - 获取类型字符串
- (NSString *)propertTypeConvert:(NSString *)typeStr
{
    NSString *resultStr = SQL_TEXT;
    if ([typeStr hasPrefix:@"T@\"NSString\""]) {
        resultStr = SQL_TEXT;
    } else if ([typeStr hasPrefix:@"T@\"NSData\""]) {
        resultStr = SQL_BLOB;
    } else if ([typeStr hasPrefix:@"Ti"]||[typeStr hasPrefix:@"TI"]||[typeStr hasPrefix:@"Ts"]||[typeStr hasPrefix:@"TS"]||[typeStr hasPrefix:@"T@\"NSNumber\""]||[typeStr hasPrefix:@"TB"]||[typeStr hasPrefix:@"Tq"]||[typeStr hasPrefix:@"TQ"]) {
        resultStr = SQL_INTEGER;
    } else if ([typeStr hasPrefix:@"Tf"] || [typeStr hasPrefix:@"Td"]){
        resultStr= SQL_REAL;
    }
    
    return resultStr;
}

#pragma mark - 得到表里的字段名称
- (NSArray *)getColumnArr:(NSString *)tableName db:(FMDatabase *)db
{
    NSMutableArray *mArr = [NSMutableArray arrayWithCapacity:0];
    FMResultSet *resultSet = [db getTableSchema:tableName];
    while ([resultSet next]) {
        [mArr addObject:[resultSet stringForColumn:@"name"]];
    }
    return mArr;
}
@end
