//
//  NSAttributedString+GMSize.m
//  Gengmei
//
//  Created by wangyang on 7/22/15.
//  Copyright (c) 2015 Wanmeichuangyi. All rights reserved.
//

#import "NSAttributedString+GMSize.h"
#import "GMFont.h"
#import "NSString+GM.h"

@implementation NSAttributedString (GMSize)

+ (NSMutableAttributedString *)string:(NSString *)string font:(UIFont *)font color:(UIColor *)color shadow:(BOOL)needShadow{
    
    if (!font) {
        NSAssert(0, @"请传递一个正常的字体参数");
    }
    
    if (!color) {
        NSAssert(0, @"请传递一个正常的字体参数");
    }
    
    if (![string isNonEmpty]) {
        return [[NSMutableAttributedString alloc] initWithString:@"" attributes:nil];;
    }
    
    NSMutableDictionary *stringAttribute = [@{NSForegroundColorAttributeName:color,
                                              NSFontAttributeName:font} mutableCopy];
    
    if (needShadow) {
        NSShadow *shadow = [NSShadow new];
        shadow.shadowOffset = CGSizeZero;
        shadow.shadowBlurRadius = 5;
        stringAttribute[NSShadowAttributeName] = shadow;
    }
    
    NSMutableAttributedString *attriString = [[NSMutableAttributedString alloc] initWithString:string attributes:stringAttribute];
    return attriString;
}

+ (NSMutableAttributedString *)attributedStringWithString:(NSString *)string fontSize:(CGFloat)size textColor:(UIColor *)color{
    return [self attributedStringWithString:string trimBothEndSpace:YES trimInnerReturnSpace:YES fontSize:size textColor:color lineSpacing:0];
}

+ (NSMutableAttributedString *)attributedStringWithString:(NSString *)string trimBothEndSpace:(BOOL)tripSpace trimInnerReturnSpace:(BOOL)trimReturn fontSize:(CGFloat)size textColor:(UIColor *)color lineSpacing:(CGFloat)spacing{
    
    NSMutableParagraphStyle *style = [[NSMutableParagraphStyle alloc] init];
    style.lineSpacing = spacing;
    style.lineBreakMode = NSLineBreakByTruncatingTail;
    
    NSString *trimString = string;
    
    if (tripSpace) {
        trimString = [string trimSpace];
    }
    
    if (trimReturn) {
        trimString = [trimString stringByReplacingOccurrencesOfString:@"\r" withString:@""];
        trimString = [trimString stringByReplacingOccurrencesOfString:@"\n" withString:@""];
        
    }
    
    if (!trimString) {
        trimString = @"";
    }
    
    NSAttributedString *attributedString = [[NSAttributedString alloc] initWithString:trimString attributes:@{NSFontAttributeName:GMFont(size), NSForegroundColorAttributeName:color, NSParagraphStyleAttributeName:style}];
    
    return [attributedString mutableCopy];
}

- (CGSize)sizeForBoundingRectSize:(CGSize)maxSize{
    
    // mutableCopy 以防修复原来的 string
    NSMutableAttributedString *string = [self mutableCopy];
    if (![string.string isNonEmpty]) {
        return CGSizeZero;
    }
    // 修改 lineBreakMode 为 NSLineBreakByWordWrapping，因为 Truncate tail 会在 boundingRectWithSize 中指定，在这里指定了，反而会导致 boundingRectWithSize 方法计算不准确
    NSMutableParagraphStyle *originStyle = [self attribute:NSParagraphStyleAttributeName atIndex:0 effectiveRange:NULL];
    originStyle.lineBreakMode = NSLineBreakByWordWrapping;
    
    CGSize size = [string boundingRectWithSize:maxSize options:NSStringDrawingUsesLineFragmentOrigin | NSStringDrawingTruncatesLastVisibleLine context:[[NSStringDrawingContext alloc] init]].size;
    return CGSizeMake(ceilf(size.width), ceilf(size.height));
}
@end

@implementation NSMutableAttributedString (GM)

- (void)addJustifiedAligment {
    if (self.length == 0) {
        return;
    }
    NSRange range = NSMakeRange(0, self.length);
    NSDictionary *dic = [self attributesAtIndex:0 effectiveRange:&range];
    NSMutableParagraphStyle *style = dic[NSParagraphStyleAttributeName];
    if (!style) {
        style = [[NSMutableParagraphStyle alloc] init];
    }
    style.alignment = NSTextAlignmentJustified;
    style.firstLineHeadIndent      = 0.001;    // important: must have a value to make alignment work

    [self addAttribute:NSBaselineOffsetAttributeName value:@0 range:range];
}

@end

@implementation NSString (Attributed)
- (NSString *)trimBothEnd {
    NSMutableString *string = [[NSMutableString alloc] initWithString:self];
    string = [string trimSpace];
    return string;
}

- (NSString *)trimInnerReturn {
    NSString *string = self;
    string = [self stringByReplacingOccurrencesOfString:@"\r" withString:@""];
    string = [self stringByReplacingOccurrencesOfString:@"\n" withString:@""];
    return string;
}

- (NSMutableAttributedString *)fontSize:(NSInteger)size {
    NSMutableAttributedString *string = [[NSMutableAttributedString alloc] initWithString:self attributes:@{NSFontAttributeName: GMFont(size)}];
    return string;
}
@end

@implementation NSMutableAttributedString (Attributed)
- (NSMutableAttributedString *)addColor:(UIColor *)color {
    [self addAttributes:@{NSForegroundColorAttributeName: color} range:NSMakeRange(0, self.length)];
    return self;
}

- (NSMutableAttributedString *)addLineSpace:(NSInteger)space {
    NSMutableParagraphStyle *style = [[NSMutableParagraphStyle alloc] init];
    style.lineSpacing = space;
    style.lineBreakMode = NSLineBreakByTruncatingTail;

    [self addAttributes:@{NSParagraphStyleAttributeName: style} range:NSMakeRange(0, self.length)];
    return self;
}

- (NSMutableAttributedString *)addShadow:(CGFloat)blur {
    NSShadow *shadow = [NSShadow new];
    shadow.shadowOffset = CGSizeZero;
    shadow.shadowBlurRadius = blur;
    [self addAttributes:@{NSShadowAttributeName: shadow} range:NSMakeRange(0, self.length)];
    return self;
}

@end