//
//  PhobosDataManager.m
//  GMCache
//
//  Created by Locus on 2020/1/21.
//

#import "PhobosDataManager.h"
#import "PhobosSendManager.h"
#import "PhobosConfig.h"
#import "PhobosUtil.h"
#import <MJExtension/MJExtension.h>
#import <mach/mach_time.h>

@implementation PhobosDataManager

static NSManagedObjectContext *Phobos_defaultContext;

+ (void)initialize {
    static dispatch_once_t onceToken;
    dispatch_once(&onceToken, ^{
        // 创建 NSManagedObjectContext，供埋点库访问CoreData库使用
        Phobos_defaultContext = [[NSManagedObjectContext alloc] initWithConcurrencyType:NSMainQueueConcurrencyType];
        NSBundle *bundle = [NSBundle bundleForClass:[self class]];
        // 获取埋点数据的实体
        NSURL *modelURL = [bundle URLForResource:@"GMPhobosData" withExtension:@"momd"];
        NSManagedObjectModel *model = [[NSManagedObjectModel alloc] initWithContentsOfURL:modelURL];
        
        // 创建持久化存储调度器
        NSPersistentStoreCoordinator *coordinator = [[NSPersistentStoreCoordinator alloc] initWithManagedObjectModel:model];
        
        // 创建并关联SQLite数据库文件，如果已经存在则不会重复创建
        NSString *dataPath = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES).lastObject;
        dataPath = [dataPath stringByAppendingFormat:@"/%@.sqlite",@"GMPhobos"];
        
        [coordinator addPersistentStoreWithType:NSSQLiteStoreType configuration:nil URL:[NSURL fileURLWithPath:dataPath] options:nil error:nil];
        
        // 设置storeCoordinator
        Phobos_defaultContext.persistentStoreCoordinator = coordinator;
        
        /** 将上次没有获取到发送结果的数据的状态修改为发送失败，待下次重新发送 */
        NSPredicate *predicate = [NSPredicate predicateWithFormat:@"status = %d", PhobosDataSendStatusSending];
        NSArray<PhobosSendDataEntity *> *entities = [self fetchDataEntitiesWithPredicate:predicate];
        [self updateDataEntities:entities sendStatus:PhobosDataSendStatusError];
        
        /** 将发送成功的数据删除 */
        NSPredicate *finishPredicate = [NSPredicate predicateWithFormat:@"status = %d", PhobosDataSendStatusFinish];
        NSArray<PhobosSendDataEntity *> *finishEntities = [self fetchDataEntitiesWithPredicate:finishPredicate];
        [self deleteDataEntities:finishEntities];
    });
}

+ (NSArray<PhobosSendDataEntity *> *)fetchToBeSendDataEntities {
    NSPredicate *predicate = [NSPredicate predicateWithFormat:@"status = %d or status = %d", PhobosDataSendStatusToBeSend, PhobosDataSendStatusError];
    return [self fetchDataEntitiesWithPredicate:predicate];
}

+ (NSArray<PhobosSendDataEntity *> *)fetchDataEntitiesWithPredicate:(NSPredicate *)searchFilter {
    return [PhobosSendDataEntity MR_findAllWithPredicate:searchFilter inContext:Phobos_defaultContext];
}

+ (NSUInteger)fetchCountOfToBeSendEntities {
    NSPredicate *predicate = [NSPredicate predicateWithFormat:@"status = %d or status = %d", PhobosDataSendStatusToBeSend, PhobosDataSendStatusError];
    return [self fetchCountOfEntitiesWithPredicate:predicate];
}

+ (NSUInteger)fetchCountOfEntitiesWithPredicate:(NSPredicate *)searchFilter {
    return [[PhobosSendDataEntity MR_numberOfEntitiesWithPredicate:searchFilter inContext:Phobos_defaultContext] integerValue];
}

+ (void)insertData:(NSDictionary *)data sendAPI:(NSString *)sendAPI {
    [self insertData:data sendAPI:sendAPI completion:nil];
}

+ (void)insertData:(NSDictionary *)data sendAPI:(NSString *)sendAPI completion:(MRSaveCompletionHandler)completion {
    if (!sendAPI || [sendAPI isEqualToString:@""] || !data) {
        return;
    }
    PhobosSendDataEntity *entity = [PhobosSendDataEntity MR_createEntityInContext:Phobos_defaultContext];
    entity.data = [data mj_JSONData];
    entity.api = sendAPI;
    entity.status = PhobosDataSendStatusToBeSend;
    entity.id = mach_absolute_time();
    [self saveWithCompletion:completion];
}

+ (void)updateDataEntities:(NSArray<PhobosSendDataEntity *> *)entities sendStatus:(PhobosDataSendStatus)sendStatus {
    [self updateDataEntities:entities sendStatus:sendStatus completion:nil];
}

+ (void)updateDataEntities:(NSArray<PhobosSendDataEntity *> *)entities sendStatus:(PhobosDataSendStatus)sendStatus completion:(MRSaveCompletionHandler)completion {
    if (entities.count > 0) {
        [entities enumerateObjectsUsingBlock:^(PhobosSendDataEntity *obj, NSUInteger idx, BOOL * _Nonnull stop) {
            obj.status = sendStatus;
        }];
        [self saveWithCompletion:completion];
    }
}

+ (void)deleteDataEntities:(NSArray<PhobosSendDataEntity *> *)entities {
    if (entities.count > 0) {
        [entities enumerateObjectsUsingBlock:^(PhobosSendDataEntity *obj, NSUInteger idx, BOOL * _Nonnull stop) {
            [obj MR_deleteEntityInContext:Phobos_defaultContext];
        }];
        [self saveWithCompletion:nil];
    }
}

+ (void)deleteAllEntities {
    [PhobosSendDataEntity MR_truncateAllInContext:Phobos_defaultContext];
    [self saveWithCompletion:nil];
}

/**
 * 在保存完成后调用的完成块。如果发生错误，块将以“BOOL”和“NSError”实例的形式传递成功状态。总是在主队列上调用。
 */
+ (void)saveWithCompletion:(MRSaveCompletionHandler)completion {
    [Phobos_defaultContext MR_saveOnlySelfWithCompletion:completion];
}

@end
