//
//  Phobos.m
//  GengmeiDoctor
//
//  Created by Thierry on 16/1/26.
//  Copyright © 2016年 wanmeizhensuo. All rights reserved.
//

#import "Phobos.h"
#import <GMCache/GMCache-umbrella.h>
#import <AdSupport/AdSupport.h>
#import "PhobosUtil.h"
#import "PhobosConfig.h"
#import "UIResponder+PhobosPV.h"
#import "PhobosUtil.h"

static Phobos *sharedClient = nil;
static NSString *sdkVersion = @"110";

@interface Phobos ()
@property (strong, nonatomic) UIViewController *visibleController;
@property (strong, nonatomic) NSDateFormatter *dateFormatter;
@property (strong, nonatomic) NSString *appName;
@property (strong, nonatomic) NSString *channelId;
@property (strong, nonatomic) NSString *appVersion;
@property (strong, nonatomic) NSString *sessionId;

@end

@implementation Phobos

+ (Phobos *)clientWithAppName:(NSString *)appName channelId:(NSString *)channelId{
    return [[self alloc] initWithAppName:appName channelId:channelId];
}

+ (Phobos *)sharedClient{
    return sharedClient;
}

+ (void)setSharedClient:(Phobos *)client{
    sharedClient = client;
    [sharedClient handleEventAfterInit];
}

- (instancetype)initWithAppName:(NSString *)appName channelId:(NSString *)channelId{
    self = [super init];
    if (self) {
        _appName = appName;
        _channelId = channelId;
        _logEnabled = NO;
        _userId = 0;
        _netStatus = @"";
        _currentCityId = @"";
        _userType = [[NSMutableDictionary alloc] initWithCapacity:0];
        _appVersion = [PhobosUtil getAppVersion];
        [self setupNotification];
        [self handleSessionStart];
        
        phobosLog(@"starts to orbit");
    }
    return self;
}

- (void)dealloc{
    if (self) {
        [[NSNotificationCenter defaultCenter] removeObserver:self];
    }
}

- (void)setUserType:(NSMutableDictionary *)userType {
    if (userType == nil && userType.count == 0) {
        return;
    }
    NSArray *newKeys = userType.allKeys;
    NSArray *oldKeys = _userType.allKeys;
    for (NSString *newKey in newKeys) {
        if ([oldKeys containsObject:newKey]) {
            NSString *newValue = [[userType objectForKey:newKey] stringValue];
            NSString *oldValue = [[_userType objectForKey:newKey] stringValue];
            if (![newValue isEqualToString:oldValue]) {
                [_userType setObject:[userType objectForKey:newKey] forKey:newKey];
            }
        } else {
            [_userType setObject:[userType objectForKey:newKey] forKey:newKey];
        }
    }
}

- (void)handleEventAfterInit{
    WMCacheService *cache = [WMCacheService sharedInstance];
    
    /** 每次打开APP埋点 **/
    [Phobos track:@"device_opened" attributes:@{} sendNow:YES];
    
    /** 第一次打开APP埋点 **/
    if (![cache fetchObjectAtDiskWithkey:PhobosHaveOpenApp]) {
        [Phobos track:@"device_activated" attributes:@{} sendNow:YES];
        [Phobos track:@"device_activated" attributes:@{} sendNow:NO];
        [cache storeObjectAtDiskWithkey:PhobosHaveOpenApp object:PhobosHaveOpenApp];
    }
}

- (UIViewController *)visibleController {
    id visibleController = [UIApplication sharedApplication].keyWindow.rootViewController;
    if ([visibleController isKindOfClass:[UITabBarController class]]) {
        UITabBarController *tabbar = (UITabBarController *)visibleController;
        UINavigationController *navigationController = (UINavigationController *)tabbar.selectedViewController;
        return navigationController.visibleViewController;
    } else if ([visibleController isKindOfClass:[UINavigationController class]]) {
        UINavigationController *navigationController = (UINavigationController *)visibleController;
        return navigationController.visibleViewController;
    } else {
        return visibleController;
    }
}

#pragma mark - notification handler
/**
 *  @brief 设置对APP的通知监听
 *
 *  @since 0.0.1
 */
- (void)setupNotification{
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(handleAppFinishLaunch:) name:UIApplicationDidFinishLaunchingNotification object:nil];
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(handleAppInForeground:) name:UIApplicationWillEnterForegroundNotification object:nil];
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(handleAppInBackgound:) name:UIApplicationDidEnterBackgroundNotification object:nil];
}

/**
 *  @brief 每次打开APP或返回前台，即Session开始的时候的处理
 *  @notification didFinishLaunch和willEnterForeground的时候都需要记录
 *
 *  @since 0.0.1
 */
- (void)handleSessionStart{
    _sessionId = [[NSUUID UUID] UUIDString];
    WMCacheService *cache = [WMCacheService sharedInstance];
    [cache storeObjectAtDiskWithkey:PhobosBeginTime object:[PhobosUtil currentTime]];
}

/**
 *  @brief 应用打开时的处理
 *
 *  @param sender
 *
 *  @since 0.0.1
 */
- (void)handleAppFinishLaunch:(id)sender{
    phobosLog(@"handleAppFinishLaunch");
    [self handleSessionStart];
}

/**
 *  @brief 应用进入前台的处理
 *
 *  @param sender
 *
 *  @since 0.0.1
 */
- (void)handleAppInForeground:(id)sender{
    phobosLog(@"handleAppInForeground");
    [self handleSessionStart];
    [self fetchDataAndSend];
    [self handlePVEventAppInForeground];
}

/**
 *  @brief 应用进入后台的处理
 *
 *  @param sender
 *
 *  @since 0.0.1
 */
- (void)handleAppInBackgound:(id)sender{
    phobosLog(@"handleAppInBackgound");
    [self handlePVEventAppInBackgound];
    [self handleSessionOver];
    [self fetchDataAndSend];
}

/**
 *  @brief 会话结束时候的处理
 *
 *  @since 0.0.1
 */
- (void)handleSessionOver{
    WMCacheService *cache = [WMCacheService sharedInstance];
    //进入后台的同时，把记录时间同步到服务端，把已使用时间清空
    double beginTime = [[cache fetchObjectAtDiskWithkey:PhobosBeginTime] doubleValue];
    if (beginTime == 0) {
        return;
    }
    NSDate *date = [NSDate date];
    double endTime = [date timeIntervalSince1970];
    NSString *usedTime = [NSString stringWithFormat:@"%ld",(long)(endTime - beginTime)];
    NSMutableDictionary *dict = [NSMutableDictionary dictionaryWithObjectsAndKeys:usedTime,@"duration",nil];
    [Phobos track:@"on_app_session_over" attributes:dict];
    [cache removeObjectAtDiskWithkey:PhobosBeginTime];
}

/**
 APP从后台到前台的时候，重新初始化pagename等信息
 @author zhaiguojun 16-10-11 in (null)
 */
- (void)handlePVEventAppInForeground {
    if (self.visibleController != nil) {
        [self onPVStart:self.visibleController];
    }
}

/**
 APP进到后台的时候，把当前pageview时间结束
 @author zhaiguojun 16-10-11 in (null)
 */
- (void)handlePVEventAppInBackgound {
    if (self.visibleController != nil) {
        [self onPVEnd:self.visibleController];
    }
}

#pragma mark - track event handler
+ (void)track:(NSString *)eventId{
    [Phobos track:eventId attributes:@{}];
}

+ (void)track:(NSString *)eventId attributes:(NSDictionary *)attributes{
    [self track:eventId attributes:attributes sendNow:NO];
    NSArray *array = [[WMCacheService sharedInstance] fetchObjectAtDiskWithkey:PhobosCacheKey];
    //超过一定数量的话，统一发送一次
    if (array.count > PhobosShardCount) {
        [sharedClient sendArray:array cleanCacheRightNow:YES];
    }
}

+ (void)track:(NSString *)eventId attributes:(NSDictionary *)attributes sendNow:(BOOL)sendNow{
    NSDictionary *dict = [sharedClient prepareDictionaryForEvent:eventId attributes:attributes];
    if (sendNow) {
        NSArray *array = @[dict];
        // 实时发送的埋点，不能立即清楚缓存
        [sharedClient sendArray:array cleanCacheRightNow:NO];
    }else{
        [sharedClient save:dict];
    }
}

+ (void)trackJsEvent:(NSString *)jsonString{
    @try {
        NSData *data = [jsonString dataUsingEncoding:NSUnicodeStringEncoding];
        NSMutableDictionary *dict = [NSJSONSerialization JSONObjectWithData:data options:NSJSONReadingAllowFragments error:nil];
        
        id pa = dict[@"params"];
        NSDictionary *json;
        if ([pa isKindOfClass:[NSString class]]) {
            NSError *jsonError;
            NSData *objectData = [pa dataUsingEncoding:NSUTF8StringEncoding];
            json = [NSJSONSerialization JSONObjectWithData:objectData
                                                   options:NSJSONReadingMutableContainers
                                                     error:&jsonError];
            [Phobos track:dict[@"type"] attributes:json];
        }else{
            [Phobos track:dict[@"type"]];
        }
    }
    @catch (NSException *exception) {
        phobosLog(exception);
    }
}

#pragma mark - PV
- (void)onPVStart:(UIResponder<PhobosPVProtocol> *)page {
    // 必须在此处调用一下referer，因为onControllerStart
    [page initReferer];
    page.inTime = [PhobosUtil currentTime];
}

- (void)onPVEnd:(UIResponder<PhobosPVProtocol> *)page {
    if (![PhobosUtil isNonEmpty:page.pageName] || !page.needLogPV) {
        return;
    }

    NSMutableDictionary *dict = [[NSMutableDictionary alloc] init];
    @try {
        [dict setObject:[PhobosUtil currentTime] forKey:@"out"];
        [dict setObject:page.inTime?:@"" forKey:@"in"];
        [dict setObject:page.pageName forKey:@"page_name"];
        [dict setObject:page.businessId?:@"" forKey:@"business_id"];
        [dict setObject:page.referer?:@"" forKey:@"referer"];
        [dict setObject:@(0) forKey:@"fake"];
        [dict setObject:page.referrerId ? : @"" forKey:@"referrer_id"];
        [Phobos track:@"page_view" attributes:dict];
    }
    @catch (NSException *exception) {
        phobosLog(exception);
    }
}

- (void)simulativePV:(NSString *)pageName businessId:(NSString *)bid referer:(NSString *)referer{
    NSMutableDictionary *dict = [[NSMutableDictionary alloc] init];
    @try {
        // fake（模拟）的事件，所以in与out一样，就是这么规定的
        [dict setObject:[PhobosUtil currentTime] forKey:@"in"];
        [dict setObject:[PhobosUtil currentTime] forKey:@"out"];
        [dict setObject:pageName?:@"" forKey:@"page_name"];
        [dict setObject:bid?:@"" forKey:@"business_id"];
        [dict setObject:referer?:@"" forKey:@"referer"];
        [dict setObject:@(1) forKey:@"fake"];
        
        [Phobos track:@"page_view" attributes:dict];
    }
    @catch (NSException *exception) {
        phobosLog(exception);
    }
}


#pragma mark - 事件存储、发送

/**
 *  @brief 将埋点时间封装成词典数据
 *
 *  @return
 *
 *  @since 0.0.1
 */
- (NSDictionary *)prepareDictionaryForEvent:(NSString *)eventId attributes:(NSDictionary *)attributes{
    [self catchNullForEvent:eventId attributes:attributes];
    NSMutableDictionary *dict = [[NSMutableDictionary alloc] init];
    @try {
        NSString *currentTime = [PhobosUtil currentTime];
        NSMutableDictionary *deviceParams = [NSMutableDictionary dictionaryWithObjectsAndKeys:
                                             [[[ASIdentifierManager sharedManager] advertisingIdentifier] UUIDString],@"idfa",
                                             [[[UIDevice currentDevice] identifierForVendor] UUIDString],@"idfv",
                                             [PhobosUtil deviceId],@"device_id",
                                             @"ios",@"device_type",
                                             @"Apple",@"manufacturer",
                                             @(self.gps.coordinate.latitude),@"lat",
                                             @(self.gps.coordinate.longitude),@"lng",
                                             _netStatus,@"is_WiFi",nil];
        NSMutableDictionary *appParams   = [NSMutableDictionary dictionaryWithObjectsAndKeys:
                                            self.appName, @"name",
                                            self.appVersion, @"version",
                                            self.channelId,@"channel",
                                            _userType,@"user_type",
                                            self.currentCityId,@"current_city_id", nil];
        [dict setObject:eventId forKey:@"type"];
        [dict setObject:appParams forKey:@"app"];
        [dict setObject:sdkVersion forKey:@"version"];
        [dict setObject:deviceParams forKey:@"device"];
        [dict setObject:@(_userId) forKey:@"user_id"];
        [dict setObject:currentTime forKey:@"create_at"];
        [dict setObject:attributes forKey:@"params"];
        [dict setObject:_sessionId forKey:@"app_session_id"];
    }
    @catch (NSException *exception) {
        phobosLog(exception);
    }
    return dict;
}

/**
 *  @brief 保存数据到缓存层
 *
 *  @param data 数据
 *
 *  @since 0.0.1
 */
- (void)save:(NSDictionary *)data
{
    if (_logEnabled) {
        phobosLog([NSString stringWithFormat:@"save dictionary: %@",data]);
    }
    WMCacheService *cache = [WMCacheService sharedInstance];
    NSMutableArray *dataArray = [cache fetchObjectAtDiskWithkey:PhobosCacheKey];
    if (dataArray) {
        [dataArray addObject:data];
    }else{
        dataArray = [NSMutableArray arrayWithObject:data];
    }
    [cache storeObjectAtDiskWithkey:PhobosCacheKey object:dataArray];
}

/**
 *  @brief 从缓存中获取数据，并发送
 *
 *  @since 0.0.1
 */
- (void)fetchDataAndSend{
    NSArray *paramsArray = [[WMCacheService sharedInstance] fetchObjectAtDiskWithkey:PhobosCacheKey];
    if (paramsArray.count>0) {
        [self sendArray:paramsArray cleanCacheRightNow:YES];
    }
}

/**
 向Mars发送埋点数据，如果是实时发送的，http请求成功之后，不清楚已有的缓存数据。
 针对普通埋点数据，发送成功之后，立即删除本地的缓存。
 @author zhaiguojun 16-10-17 in (null)
 @param array 参数
 @param now   是否立即清楚缓存
 */
- (void)sendArray:(NSArray *)array cleanCacheRightNow:(BOOL)clean {
    if (_logEnabled) {
        phobosLog([NSString stringWithFormat:@"array prepare to fly --✈: %@",array]);
    }
    @try {
        NSData *JSON = [PhobosUtil encodeJSON:array];
        NSData *compressedData = [PhobosUtil compressData:JSON];
        if (compressedData) {
            [PhobosUtil sendData:compressedData success:^(NSInteger code) {
                phobosLog(@"✈ ---------- ✈ data arrived Mars");
                if (clean) {
                    [[WMCacheService sharedInstance] removeObjectAtDiskWithkey:PhobosCacheKey];
                }
            }];
        }
    }
    @catch (NSException *exception) {
        phobosLog(exception);
    }
}

#pragma mark - helpers
- (void)catchNullForEvent:(NSString *)eventId attributes:(NSDictionary *)attributes {
    dispatch_async(dispatch_get_global_queue(0, 0), ^{
        @try {
            for (NSString *key in attributes.allKeys) {
                if ([attributes[key] isMemberOfClass:[NSNull class]]) {
                    if (self.captureNullExpection) {
                        self.captureNullExpection(eventId, attributes);
                    }
                    break;
                }
            }
        } @catch (NSException *exception) {
            
        }
    });
}

@end
