//
//  GMExposureManager.m
//  Gengmei
//
//  Created by Mikasa on 2019/6/21.
//  Copyright © 2019 更美互动信息科技有限公司. All rights reserved.
//

#import "GMExposureManager.h"
#import "GMExposureModel.h"
#import "UIResponder+PhobosPV.h"
#import "Phobos.h"
#import "PhobosUtil.h"
#import <MJExtension/MJExtension.h>

@interface GMExposureManager ()
/// 曝光页面
@property (nonatomic, strong) NSMutableArray *pageArrayM;
@property (nonatomic, strong) NSMutableDictionary *exposureDictM;// 存储开始进入曝光的数据
@property (nonatomic, strong) NSMutableDictionary *exposureViewDictM;// 存储开始进入曝光的View
@property (nonatomic, strong) NSMutableDictionary *exposurePageDictM;//存储页面上已经曝光过的View

@end
@implementation GMExposureManager

#pragma mark - init Method
+ (instancetype)sharedManager {
    static dispatch_once_t onceToken;
    static id _instance;
    dispatch_once(&onceToken, ^{
        _instance = [[GMExposureManager alloc] init];
    });
    return _instance;
}

- (instancetype)init {
    
    if (self =[super init]) {
        
        _pageArrayM = [NSMutableArray array];
        _exposureDictM = [NSMutableDictionary dictionary];
        _exposurePageDictM = [NSMutableDictionary dictionary];
        _exposureViewDictM = [NSMutableDictionary dictionary];
        _uploadMode = GMExpoUploadModePage;
        _exposureDimThreshold = 0.5f;
        _exposureMinDuration = 0.01f;
        [self addObserverMethod];
    }
    return self;
}

#pragma mark - Observer Method

- (void)addObserverMethod {
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(handleNotificaion:) name:UIApplicationDidEnterBackgroundNotification object:nil];
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(handleNotificaion:) name:UIApplicationWillEnterForegroundNotification object:nil];
}

- (void)removeObserverMethod {
    [[NSNotificationCenter defaultCenter] removeObserver:self
                                                    name:UIApplicationDidEnterBackgroundNotification object:nil];
    [[NSNotificationCenter defaultCenter] removeObserver:self
                                                    name:UIApplicationWillEnterForegroundNotification object:nil];
}

// 处理通知事件
- (void)handleNotificaion:(NSNotification *)notific {
    
    NSString *notifName = notific.name;
    
    // 进入后台 结束曝光 + 上报所有数据
    if ([notifName isEqualToString:UIApplicationDidEnterBackgroundNotification] ) {
        
        [GMExposureManager sharedManager].inBackground = YES;
        
        // 同步所有页面数据并上传
        [[GMExposureManager sharedManager] endExpoTrcakerForAllPage];
    }
    
    // 回到前台 记录数据
    if ([notifName isEqualToString:UIApplicationWillEnterForegroundNotification] ) {
        [GMExposureManager sharedManager].inBackground = NO;
        
        // 取出临时存储的需要曝光的页面，进行曝光
        for (NSString *pageKey in self.pageArrayM) {
            UIViewController *pageCtrl = [GMExposureManager expoPageCtrl:pageKey];
            if (pageCtrl && [pageCtrl isKindOfClass:[UIViewController class]]) {
                [GMExposureManager fetchViewForVisibleState:pageCtrl.view trackerType:GMViewTrackerTypeUIViewEnterForeground recursive:YES];
            }
        }
        [self.pageArrayM removeAllObjects];
    }
}

#pragma mark - Tracker Method

// 结束所有页面的曝光并上报
- (void)endExpoTrcakerForAllPage {
    
    // 将所有正在曝光的View的状态变更为结束曝光
    for (UIView *view in self.exposureViewDictM.allValues) {
        view.expoStatus = GMViewExpoStatusEnd;
    }
    [self.exposureViewDictM removeAllObjects];
    
    for (NSString *viewKey in self.exposureDictM.allKeys) {
        
        // 将正在曝光的数据直接更改状态为结束曝光
        NSDictionary *dict = self.exposureDictM[viewKey];
        NSString *page = [[viewKey componentsSeparatedByString:@"/"] firstObject];
        UIViewController *pageCtrl = [GMExposureManager expoPageCtrl:page];
        [self storeExposureData:dict pageCtrl:pageCtrl];
    }
    [self.exposureDictM removeAllObjects];
    
    // 将之前曝光的页面临时存储起来，确保下次从后台回来页面会立即开启曝光
    [self.pageArrayM addObjectsFromArray:self.exposurePageDictM.allKeys];
    
    [self uploadAllExposureData];
}

/**
 1.结束指定页面的曝光并上报
 2.多用于下拉刷新前需要曝光当前的vc的数据
 @param pageCtrl 指定页面
 */
- (void)endExpoTrcakerForPageCtrl:(UIViewController *)pageCtrl {
    
    // 在结束之前获取一次
    [GMExposureManager fetchViewForVisibleState:pageCtrl.view trackerType:GMViewTrackerTypeUIViewDidDisappear recursive:YES];

    NSString *page = [NSString stringWithFormat:@"%p",pageCtrl];
    // View
    NSMutableDictionary *dictViewM = [NSMutableDictionary dictionaryWithDictionary:self.exposureViewDictM];
    for (NSString *viewKey in self.exposureViewDictM.allKeys) {
        if ([viewKey hasPrefix:page]) {
            UIView *view = self.exposureViewDictM[viewKey];
            view.expoStatus = GMViewExpoStatusEnd;
            [dictViewM removeObjectForKey:viewKey];
        }
    }
    self.exposureViewDictM = dictViewM;
    
    // data
    NSMutableDictionary *dictDataM = [NSMutableDictionary dictionaryWithDictionary:self.exposureDictM];
    for (NSString *viewKey in self.exposureDictM.allKeys) {
        if ([viewKey hasPrefix:page]) {
            NSDictionary *dict = self.exposureDictM[viewKey];
            [self storeExposureData:dict pageCtrl:pageCtrl];
            [dictDataM removeObjectForKey:viewKey];
        }
    }
    self.exposureDictM = dictDataM;
    
    [self uploadExposureDataForPage:pageCtrl];;
}

#pragma mark - Deal Data
// 存储曝光数据
- (void)startStoreExposureData {
    
    // 当前曝光View 上数据曝光情况
    NSMutableDictionary *dictDataM = [NSMutableDictionary dictionaryWithDictionary:self.exposureDictM];
    for (NSString *key in dictDataM.allKeys) {
        
        UIView *exposureView = self.exposureViewDictM[key];
        
        // 当前view已经结束曝光 记录曝光数据---等待上传
        if (!exposureView) {
            NSString *page = [[key componentsSeparatedByString:@"/"] firstObject];
            UIViewController *pageCtrl = [GMExposureManager expoPageCtrl:page];
            [self storeExposureData:dictDataM[key] pageCtrl:pageCtrl];
            [self.exposureDictM removeObjectForKey:key];
        }
    }
    
    // 当前曝光数据是否仍在曝光-> 无存储曝光数据
    NSMutableDictionary *dictViewM = [NSMutableDictionary dictionaryWithDictionary:self.exposureViewDictM];
    for (NSString *key in dictViewM.allKeys) {
        NSDictionary *dataDict = self.exposureDictM[key];
        if (!dataDict) {
            UIView *view = dictViewM[key];
            [self storeStartExposureData:view pageCtrl:view.pageCtrl];
        }
    }
}

// 存储开始曝光数据
- (void)storeStartExposureData:(UIView *)startView pageCtrl:(UIViewController *)pageCtrl {
    
    NSMutableDictionary *exposureDictM = [GMExposureManager sharedManager].exposureDictM;
    startView.inTime = [PhobosUtil currentTime];
    startView.expoStatus = GMViewExpoStatusStart;
    
    NSMutableDictionary *dict = [NSMutableDictionary dictionaryWithDictionary:startView.exposure];
    // 安卓没有这个功能 暂时屏蔽
    //[dict setObject:SafeString([PhobosUtil currentTime]) forKey:@"inTime"];
    
    // 获取卡片位置
    [dict setObject:startView.relative_position?:@"" forKey:@"relative_position"];
    [dict setObject:startView.absolute_position?:@"" forKey:@"absolute_position"];
    
    if (startView.extra_param && startView.extra_param.allKeys.count > 0) {
        [dict addEntriesFromDictionary:startView.extra_param];
    }
    [exposureDictM setObject:dict forKey:[GMExposureManager exposureKeyForData:startView]];
}

/**
 按指定页面存储曝光数据
 
 @param view 曝光数据
 @param pageName 指定页面
 */
- (void)storeExposureData:(NSDictionary *)expoDict pageCtrl:(UIViewController *)pageCtrl {
    
    // 卡片曝光持续时间超出曝光最少时长才进行记录
    /*
    NSInteger inTime = [expoDict[@"inTime"] integerValue];
    NSInteger duration = [view.outTime integerValue] - [view.inTime integerValue];
    if (!duration) {
        return;
    }*/
    
    // 将曝光数据所需要的某些参数从Page中取出
    NSString *pageKey = [GMExposureManager exposureKeyForPage:pageCtrl];
    GMExposureItemModel *itemModel = [self.exposurePageDictM objectForKey:pageKey];
    if (!itemModel) {
        itemModel = [GMExposureItemModel new];
    }
    
    NSMutableDictionary *dict = [NSMutableDictionary dictionaryWithDictionary:expoDict];
    /* 安卓没有此功能 暂屏蔽
    if (![dict[@"outTime"] isNonEmpty]) {
        [dict setObject:SafeString([PhobosUtil currentTime]) forKey:@"outTime"];
    }*/
    [itemModel.exposure_cards addObject:dict];
    [self.exposurePageDictM setObject:itemModel forKey:pageKey];
}

// 同步Item数据
- (void)synchronizeViewItemData:(UIView *)view {
    
    NSMutableDictionary *dict = [self.exposureDictM objectForKey:[GMExposureManager exposureKeyForData:view]];
    if (dict) {
        
        if (![PhobosUtil isNonEmpty:dict[@"relative_position"]]|| ![PhobosUtil isNonEmpty:dict[@"absolute_position"]]) {
            [view fechExposurePositionForView];
            [dict setObject:view.relative_position?:@"" forKey:@"relative_position"];
            [dict setObject:view.absolute_position?:@"" forKey:@"absolute_position"];
            if (view.extra_param && view.extra_param.allKeys.count > 0) {
                [dict addEntriesFromDictionary:view.extra_param];
            }
            [self.exposureDictM setObject:dict forKey:[GMExposureManager exposureKeyForData:view]];
        }
    }
}

// 进行数据同步
- (void)synchronizeUploadItem:(GMExposureItemModel *)itemModel withViewPageData:(UIViewController *)pageCtrl {
    
    if (!itemModel || !pageCtrl) {
        return;
    }
    
    itemModel.tab_name = pageCtrl.tabName?:@"";
    itemModel.page_name = pageCtrl.pageName?:@"";
    itemModel.business_id = pageCtrl.businessId?:@"";
    itemModel.referrer = pageCtrl.referer?:@"";
    itemModel.referrerId = pageCtrl.referrerId?:@"";
    itemModel.up_slide_times = pageCtrl.up_slide_times;
    itemModel.down_slide_times = pageCtrl.down_slide_times;
    itemModel.up_loading_times = pageCtrl.up_loading_times;
    itemModel.down_loading_times = pageCtrl.down_loading_times;
    
    [itemModel.exposure_cards sortUsingComparator:^NSComparisonResult(NSDictionary *obj1, NSDictionary *obj2){
        return [obj1[@"absolute_position"] integerValue] > [obj2[@"absolute_position"] integerValue];
    }];
    
    NSDictionary *dic = [itemModel mj_keyValues];
    
    NSMutableDictionary *dicM = [NSMutableDictionary dictionaryWithDictionary:dic];
    [dicM addEntriesFromDictionary:pageCtrl.extra_param];
    
    [Phobos track:@"page_precise_exposure" attributes:dicM sendNow:YES currentAPI:self.uploadExposureAPI];
}

/**
 上传指定页面的曝光数据
 
 @param pageName 指定页面
 */
- (void)uploadExposureDataForPage:(UIViewController *)pageCtrl {
    
    GMExposureItemModel *itemModel = [self.exposurePageDictM objectForKey:[GMExposureManager exposureKeyForPage:pageCtrl]];
    if (!itemModel.exposure_cards || itemModel.exposure_cards.count == 0) {
        return;
    }
    [self synchronizeUploadItem:itemModel withViewPageData:pageCtrl];
    // 上传成功后进行数据清理
    [self clearExposureDataForPage:pageCtrl];
}

/**
 上传所有页面的曝光数据
 */
- (void)uploadAllExposureData {
    
    // 同步所有页面数据并上传
    __weak __typeof(self)weakSelf = self;
    NSDictionary *pageDictM = [GMExposureManager sharedManager].exposurePageDictM;
    [pageDictM enumerateKeysAndObjectsUsingBlock:^(id  _Nonnull key, id  _Nonnull obj, BOOL * _Nonnull stop) {
        
        NSString *pageName = key;
        UIViewController *pageCtrl = [GMExposureManager expoPageCtrl:pageName];
        
        [weakSelf uploadExposureDataForPage:pageCtrl];
        
    }];
    [self clearAllExposureData];
}

/**
 清理指定页面的曝光数据
 
 @param pageName 指定页面
 */
- (void)clearExposureDataForPage:(UIViewController *)pageCtrl {
    
    pageCtrl.up_loading_times = 0;
    pageCtrl.down_loading_times = 0;
    pageCtrl.up_slide_times = 0;
    pageCtrl.down_slide_times = 0;
    pageCtrl.extra_param = [NSDictionary dictionary];
    
    NSString *pageName = [GMExposureManager exposureKeyForPage:pageCtrl];
    
    // 清理指定页面存储view
    NSMutableDictionary *dictViewM = [NSMutableDictionary dictionaryWithDictionary:self.exposureViewDictM];
    for (NSString *viewKey in self.exposureViewDictM.allKeys) {
        if ([viewKey hasPrefix:pageName]) {
            [dictViewM removeObjectForKey:viewKey];
        }
    }
    self.exposureViewDictM = dictViewM;
    
    // 清理指定页面的数据
    NSMutableDictionary *dicDataM = [NSMutableDictionary dictionaryWithDictionary:self.exposureDictM];
    for (NSString *viewKey in self.exposureDictM.allKeys) {
        if ([viewKey hasPrefix:pageName]) {
            [dicDataM removeObjectForKey:viewKey];
        }
    }
    [GMExposureManager sharedManager].exposureDictM = dicDataM;
    
    [[GMExposureManager sharedManager].exposurePageDictM removeObjectForKey:pageName];
}

/**
 清理所有页面的曝光数据
 */
- (void)clearAllExposureData {
    
    [[GMExposureManager sharedManager].exposurePageDictM removeAllObjects];
    [[GMExposureManager sharedManager].exposureDictM removeAllObjects];
    [[GMExposureManager sharedManager].exposureViewDictM removeAllObjects];
}

#pragma mark - Verify ViewExposure

/**
 当前View上是否有需要精准曝光的数据
 
 @param view View
 @return yes/no
 */
+ (BOOL)haveExposureForView:(UIView *)view {
    
    if (view.exposure && view.exposure.allKeys.count > 0) {
        return YES;
    }
    return NO;
}

/**
 是否是需要精准曝光的View
 
 @param view view
 @return yes/no
 */
+ (BOOL)isTargetViewForExposure:(UIView *)view {
    
    //    NSLog(@"%@--%@",view,view.pageCtrl);
    return view.pageCtrl.needExpo;
}

/**
 是否需要忽略UITrackingRunLoopMode模式下的记录
 
 @param ignoreTracking 是否需要忽略
 @return 模式下的记录
 */
+ (BOOL)isNeedIgnoreTrackExposure:(BOOL)ignoreTracking {
    
    NSRunLoopMode mode = [NSRunLoop currentRunLoop].currentMode;
    if (ignoreTracking && [mode isEqualToString:UITrackingRunLoopMode]) {
        return YES;
    }
    return NO;
}

/**
 设置当前view是否可见状态
 
 @param view view
 @param recursive 是否递归子view
 */
+ (void)fetchViewForVisibleState:(UIView *)view  trackerType:(GMViewTrackerType)trackerType recursive:(BOOL)recursive {
    
    BOOL ignoreTracking = (trackerType == GMViewTrackerTypeUIScrollDragging || trackerType == GMViewTrackerTypeUIViewEnterForeground || trackerType == GMViewTrackerTypeUIViewDidAppear || trackerType == GMViewTrackerTypeUIViewDidDisappear || trackerType == GMViewTrackerTypeUIViewWillResignActive)?NO:YES;
    
    if ([GMExposureManager haveExposureForView:view] && [GMExposureManager isTargetViewForExposure:view]) {
        
        view.visibleView = [self isViewVisible:view];
        [view fechExposurePositionForView];
        if (![GMExposureManager isNeedIgnoreTrackExposure:ignoreTracking]) {
            /*
            if (trackerType == GMViewTrackerTypeUIViewSetExposure &&
                ![GMExposureManager sharedManager].inBackground) {
                view.visibleView = YES;
            }*/
            [[GMExposureManager sharedManager] exposureStatusForView:view inPageCtrl:view.pageCtrl];
        } else {
            
            [[GMExposureManager sharedManager] synchronizeViewItemData:view];
        }
    }
    
    if (recursive) {
        for (UIView *subview in view.subviews) {
            if (!subview.pageCtrl) {
                subview.pageCtrl = view.pageCtrl;
            }
            [GMExposureManager fetchViewForVisibleState:subview trackerType:trackerType recursive:recursive];
        }
    }
    
    if (![GMExposureManager isNeedIgnoreTrackExposure:ignoreTracking] && (!recursive || view.subviews.count == 0)) {
        [[GMExposureManager sharedManager] startStoreExposureData];
    }
}

/**
 判断当前view是否可见
 
 @param view view
 @return yes/no
 */
+ (BOOL)isViewVisible:(UIView *)view {
    
    // NSLog(@"%@:%@:%d:%d:%f",view,view.window ,view.hidden,view.layer.hidden,view.alpha);
    
    // 可见
    if (!view.window || view.hidden || view.layer.hidden || !view.alpha) {
        
        return NO;
    }
    
    // App进入后台，当前view均置为不可见
    if ([GMExposureManager sharedManager].inBackground) {
        return NO;
    }
    
    UIView *current = view;
    while ([current isKindOfClass:[UIView class]]) {
        if (current.alpha <= 0 || current.hidden) {
            return NO;
        }
        current = current.superview;
    }
    
    // 判断当前View 与 view所在window是否存在交集
    CGRect viewRectInWindow = [view convertRect:view.bounds toView:view.window];
    BOOL isIntersects = CGRectIntersectsRect(view.window.bounds, viewRectInWindow);
    
    if (isIntersects) {
        // 获取View 与 view所在window的相交区域
        CGRect intersectRect = CGRectIntersection(view.window.bounds, viewRectInWindow);
        if (intersectRect.size.width != 0.f && intersectRect.size.height != 0.f) {
            // size > exposureDimThreshold 视为可见
            CGFloat dimThreshold = [GMExposureManager sharedManager].exposureDimThreshold;
            if (intersectRect.size.width / viewRectInWindow.size.width > dimThreshold &&
                intersectRect.size.height / viewRectInWindow.size.height > dimThreshold) {
                return YES;
            }
        }
    }
    return NO;
}

#pragma mark - View Visible/UnVisibel

/** 设置view的曝光状态
 * view 可见 && 状态none或者end——> 开始曝光
 * view 可见 && 状态start——> 正在曝光,不做处理
 * view 不可见 && 状态start ——> 结束曝光
 * view 不可见 && 状态none或者end ——> 不做处理
 */
- (void)exposureStatusForView:(UIView *)view inPageCtrl:(UIViewController *)inPageCtrl {
    
    // GMViewExpoStatus status = view.expoStatus;
    if (view.visibleView) {
        // 开始曝光
        [self view:view startVisibleInPageCtrl:inPageCtrl];
    } else {
        // 结束曝光
        [self view:view endVisibleInPageCtrl:inPageCtrl];
    }
}

/**
 view 在当前页面开始可见（开始曝光）
 
 @param view 目标view
 @param inPageCtrl inPageViewContrl 当前页面
 */
- (void)view:(UIView *)startView startVisibleInPageCtrl:(UIViewController *)inPageCtrl {
    
    // 无状态进入页面直接开启曝光
    NSString *viewKey = [GMExposureManager exposureKeyForData:startView];
    UIView *view = [self.exposureViewDictM objectForKey:viewKey];
    if (!view) {
        startView.expoStatus = GMViewExpoStatusStart;
        [self.exposureViewDictM setObject:startView forKey:viewKey];
    }
}


/**
 view在当前页面结束可见（曝光结束）
 
 @param view 目标view
 @param inPageViewContrl 指定页面
 */
- (void)view:(UIView *)endView endVisibleInPageCtrl:(UIViewController *)inPageCtrl {
    
    if ([self.exposureViewDictM.allValues containsObject:endView]) {
        NSArray *keys = [self.exposureViewDictM allKeysForObject:endView];
        for (NSString *key in keys) {
            [self.exposureViewDictM removeObjectForKey:key];
        }
    }
}

#pragma mark - ExposureKey
/**
 * 获取曝光View存储Key（pageCtrl地址/inTime/view）
 * pageCtrl地址:当前view所在页面的地址,方便后续根据页面就能删除
 * inTime：当前开始曝光时间
 * view地址：view
 * /：后续可根据/进行切分
 @param view View
 @return 当前曝光View pageCtrl地址/inTime/view的地址为key
 */
+ (NSString *)exposureKeyForData:(UIView *)view {
    return [NSString stringWithFormat:@"%p/%p",view.pageCtrl,view.exposure];
}

/**
 获取页面key存储的曝光数据
 
 @param pageCtrl pageCtrl
 @return 当前页面地址
 */
+ (NSString *)exposureKeyForPage:(UIViewController *)pageCtrl {
    return [NSString stringWithFormat:@"%p",pageCtrl];
}

/// 根据view的地址获取当前的page地址
/// @param viewKey viewKey
+ (UIViewController *)expoViewKeyForPageCtrl:(NSString *)viewKey {
    
    NSString *page = [[viewKey componentsSeparatedByString:@"/"] firstObject];
    return [GMExposureManager expoPageCtrl:page];
}

/**
 根据页面地址获取页面对象
 
 @param pageAddress 页面地址
 @return 页面对象
 */
+ (UIViewController *)expoPageCtrl:(NSString *)pageAddress {
    uintptr_t hex = strtoull(pageAddress.UTF8String, NULL, 0);
    id gotcha = (__bridge id)(void *)hex;
    UIViewController *pageCtrl = (UIViewController *)gotcha;
    return pageCtrl;
    
}

- (void)dealloc {
    [self removeObserverMethod];
}

@end

