module Pod
  class Installer
    # Controller class responsible of creating and configuring the static
    # library target in Pods project. It also creates the support file needed
    # by the target.
    #
    class TargetInstaller
      # @return [Sandbox] sandbox the sandbox where the support files should
      #         be generated.
      #
      attr_reader :sandbox

      # @return [Target] The library whose target needs to be generated.
      #
      attr_reader :target

      # @param  [Project] project @see project
      # @param  [Target]  target  @see target
      #
      def initialize(sandbox, target)
        @sandbox = sandbox
        @target = target
      end

      private

      #-----------------------------------------------------------------------#

      # @!group Installation steps

      # Adds the target for the library to the Pods project with the
      # appropriate build configurations.
      #
      # @note   The `PODS_HEADERS_SEARCH_PATHS` overrides the xcconfig.
      #
      # @return [void]
      #
      def add_target
        name = target.label
        platform = target.platform.name
        deployment_target = target.platform.deployment_target.to_s
        @native_target = project.new_target(:static_library, name, platform, deployment_target)

        target.user_build_configurations.each do |bc_name, type|
          configuration = @native_target.add_build_configuration(bc_name, type)
        end

        settings = { 'OTHER_LDFLAGS' => '', 'OTHER_LIBTOOLFLAGS' => '' }
        if target.archs
          settings['ARCHS'] = target.archs
        end

        @native_target.build_configurations.each do |configuration|
          configuration.build_settings.merge!(settings)
        end

        target.native_target = @native_target
      end

      # Creates the directory where to store the support files of the target.
      #
      def create_support_files_dir
        target.support_files_dir.mkdir
      end

      # Generates a dummy source file for each target so libraries that contain
      # only categories build.
      #
      # @return [void]
      #
      def create_dummy_source
        path = target.dummy_source_path
        generator = Generator::DummySource.new(target.label)
        generator.save_as(path)
        file_reference = add_file_to_support_group(path)
        native_target.source_build_phase.add_file_reference(file_reference)
      end

      # @return [PBXNativeTarget] the target generated by the installation
      #         process.
      #
      # @note   Generated by the {#add_target} step.
      #
      attr_reader :native_target

      private

      #-----------------------------------------------------------------------#

      # @!group Private helpers.

      # @return [Project] the Pods project of the sandbox.
      #
      def project
        sandbox.project
      end

      # @return [TargetDefinition] the target definition of the library.
      #
      def target_definition
        target.target_definition
      end

      # @return [PBXGroup] the group where the file references to the support
      #         files should be stored.
      #
      attr_reader :support_files_group

      # Adds a reference to the given file in the support group of this target.
      #
      # @param  [Pathname] path
      #         The path of the file to which the reference should be added.
      #
      # @return [PBXFileReference] the file reference of the added file.
      #
      def add_file_to_support_group(path)
        support_files_group.new_file(path)
      end

      #-----------------------------------------------------------------------#
    end
  end
end
