module Pod

  # Model class which describes a Pods target.
  #
  # The Target class stores and provides the information necessary for
  # working with a target in the Podfile and it's dependent libraries.
  # This class is used to represent both the targets and their libraries.
  #
  class Target

    # @return [PBXNativeTarget] the target definition of the Podfile that
    #         generated this target.
    #
    attr_reader :target_definition

    # @return [Sandbox] The sandbox where the Pods should be installed.
    #
    attr_reader :sandbox

    # @return [String] the name of the library.
    #
    def name
      label
    end

    # @return [String] the name of the library.
    #
    def product_name
      "lib#{label}.a"
    end

    # @return [String] the XCConfig namespaced prefix.
    #
    def xcconfig_prefix
      label.upcase.gsub(/[^A-Z]/, '_') + '_'
    end

    def skip_installation?
      false
    end

    # @return [String] A string suitable for debugging.
    #
    def inspect
      "<#{self.class} name=#{name}>"
    end

    def to_s
      "#{name} #{platform}"
    end

    #-------------------------------------------------------------------------#

    # @!group Information storage

    # @return [Hash{String=>Symbol}] A hash representing the user build
    #         configurations where each key corresponds to the name of a
    #         configuration and its value to its type (`:debug` or `:release`).
    #
    attr_accessor :user_build_configurations

    # @return [PBXNativeTarget] the target generated in the Pods project for
    #         this library.
    #
    attr_accessor :target
    alias :native_target :target
    alias :native_target= :target=

    # @return [Platform] the platform for this library.
    #
    def platform
      @platform ||= target_definition.platform
    end


    # @return [Xcodeproj::Config] The configuration file of the target.
    #
    # @note   The configuration is generated by the {TargetInstaller} and
    #         used by {UserProjectIntegrator} to check for any overridden
    #         values.
    #
    attr_accessor :xcconfig

    # @return [Pathname] The path of the xcconfig file relative to the root of
    #         the user project.
    #
    attr_accessor :xcconfig_path

    # @return [Pathname] The path of the copy resources script relative to the
    #         root of the user project.
    #
    attr_accessor :copy_resources_script_path


    # @return [Pathname]
    #
    attr_accessor :prefix_header_path

    #-------------------------------------------------------------------------#

    # @!group Support files

    # @return [Pathname] the folder where to store the support files of this
    #         library.
    #
    def support_files_root
      @sandbox.library_support_files_dir(name)
    end

    #-------------------------------------------------------------------------#

  end
end
