
module Pod
  class DocsGenerator
    def self.appledoc_installed?
      !`which appledoc`.strip.empty?
    end

    include Config::Mixin

    attr_reader :pod, :specification, :target_path, :options

    def initialize(pod)
      @pod = pod
      @specification = pod.specification
      @target_path = pod.sandbox.root + "Documentation" + pod.name
      @options = pod.specification.documentation || {}
    end

    def name
      @specification.to_s
    end

    def company
      if @specification.authors
        @specification.authors.keys.join(', ')
      else
        'no-company'
      end
    end

    def copyright
      company
    end

    def description
      @specification.description || "Generated by CocoaPods."
    end

    def docs_id
      'org.cocoapods'
    end

    def files
      @pod.absolute_source_files
    end

    def index_file
      candidates = ['README.md', 'README.mdown', 'README.markdown','README']
      candidates.each do |f|
        if File.file?(@pod.root + f)
          return f
        end
      end
      nil
    end

    def spec_appledoc_options
      @options[:appledoc] || []
    end

    def generate_appledoc_options
      options = ['--project-name', name,
                 '--docset-desc', description,
                 '--project-company', company,
                 '--docset-copyright', copyright,
                 '--company-id', docs_id,
                 '--ignore', '.m',
                 '--keep-undocumented-objects',
                 '--keep-undocumented-members']
      index = index_file
      options += ['--index-desc', index] if index
      options += spec_appledoc_options
    end

    def generate(install = false)
      unless self.class.appledoc_installed?
        puts "[!] Skipping documentation generation because appledoc can't be found." if config.verbose?
        return
      end
      options = generate_appledoc_options
      options += ['--output', @target_path.to_s]
      options += ['--keep-intermediate-files']
      options += install ? ['-create-docset'] : ['--no-create-docset']
      @target_path.mkpath
      @pod.chdir do
        appledoc(options)
      end
    end

    def appledoc(options)
      arguments = []
      arguments += options
      arguments += ['--print-settings'] if config.verbose?
      arguments += files.map(&:to_s)
      Open3.popen3('appledoc', *arguments) do |i, o, e|
        if config.verbose?
          puts o.read.chomp
          puts e.read.chomp
        else
          # TODO: This is needed otherwise appledoc may not install the doc set
          # This is a work around related to poor understanding of the IO class.
          #
          # I think we can use the non-block version here, which should read
          # everything till the end and then return.
          o.read
          e.read
        end
      end
    end

  end
end

