module Pod
  class Installer

    # Controller class responsible of creating and configuring the static
    # library target in Pods project. Every target is generated from a target
    # definition of the Podfile.
    #
    class TargetInstaller

      # @param  [Project]           project @see project
      # @param  [TargetDefinition]  target_definition @see target_definition
      #
      def initialize(sandbox, library)
        @sandbox = sandbox
        @library = library
      end

      # Creates the target in the Pods project and its support files.
      #
      # @return [void]
      #
      def install!
        add_file_reference_for_support_files
        add_target
        add_build_files_to_target

        create_xcconfig_file
        create_prefix_header
        create_bridge_support_file
        create_copy_resources_script
      end

      #-----------------------------------------------------------------------#

      # @!group API
      #
      # This is the tentative API for the podfile and the specification hooks.

      public

      # @return [Sandbox] sandbox the sandbox where the support files should
      #         be generated.
      #
      attr_reader :sandbox

      # @return [Library] The library whose target needs to be generated.
      #
      attr_reader :library

      # @return [PBXNativeTarget] the target generated by the installation
      #         process.
      #
      # @note   Generated by the {#add_target} step.
      #
      attr_reader :target

      #--------------------------------------#

      # @!group Hooks compatiblity

      # @todo This has to be removed, but this means the specs have to be
      #       updated if they need a reference to the prefix header.
      #
      def prefix_header_filename
        UI.warn "The usage of the TargetInstaller#prefix_header_filename is deprecated."
        library.prefix_header_name
      end

      #-----------------------------------------------------------------------#

      # @!group Installation steps

      private

      # Adds the file references for the support files that are generated by
      # the installation process to the Pods project.
      #
      # @return [void]
      #
      def add_file_reference_for_support_files
        g = project.support_files_group.new_group(target_definition.label)
        g.new_file(library.copy_resources_script_name)
        g.new_file(library.prefix_header_name)
        @xcconfig_file_ref = g.new_file(library.xcconfig_name)
      end

      # Adds the target for the library to the Pods project with the
      # appropriate build configurations.
      #
      # @note   The `PODS_HEADERS_SEARCH_PATHS` overrides the xcconfig.
      #
      # @todo   Should we add the build configurations only to the target?
      #
      # @return [void]
      #
      def add_target
        name     = library.label
        platform = library.platform.name
        settings = build_settings_for_platform(library.platform)
        settings['GCC_PREFIX_HEADER'] = library.prefix_header_name
        if target_definition.inhibit_all_warnings?
         settings['GCC_WARN_INHIBIT_ALL_WARNINGS'] = 'YES'
        end
        Generator::XCConfig.pods_project_settings.each do |key, value|
          settings[key] = value
        end

        @target = project.new_target(:static_library, name, platform)
        @target.build_settings('Debug').merge!(settings)
        @target.build_settings('Release').merge!(settings)

        library.user_build_configurations.each do |name, type|
          unless @target.build_configurations.map(&:name).include?(name)
            build_config = project.new(Xcodeproj::Project::XCBuildConfiguration)
            build_config.name = name
            settings = @target.build_settings(type.to_s.capitalize)
            build_config.build_settings = settings
            target.build_configurations << build_config
            project.build_configurations << build_config
          end
        end

        @target.build_configurations.each do |c|
          c.base_configuration_reference = xcconfig_file_ref
        end

        library.target = @target
      end

      # Adds the build files of the pods to the target.
      #
      # @return [void]
      #
      def add_build_files_to_target
        pods.each { |p| p.add_build_files_to_target(target) }
      end

      # Generates the contents of the xcconfig file and saves it to disk.
      #
      # @note   The `ALWAYS_SEARCH_USER_PATHS` flag is enabled to support
      #         libraries like `EmbedReader`.
      #
      # @return [void]
      #
      def create_xcconfig_file
        UI.message "- Generating xcconfig file at #{UI.path(library.xcconfig_path)}" do
          gen = Generator::XCConfig.new(sandbox, pods, library.relative_pods_root)
          gen.set_arc_compatibility_flag = target_definition.podfile.set_arc_compatibility_flag?
          gen.save_as(library.xcconfig_path)
          library.xcconfig = gen.xcconfig
        end
      end

      # Creates a prefix header file which imports `UIKit` or `Cocoa`. This
      # file also include any prefix header content reported by the
      # specification of the pods.
      #
      # @return [void]
      #
      def create_prefix_header
        UI.message "- Generating prefix header at #{UI.path(library.prefix_header_path)}" do
          gen = Generator::PrefixHeader.new(target_definition.platform, pods)
          gen.save_as(library.prefix_header_path)
        end
      end

      # Generates the bridge support metadata if requested by the {Podfile}.
      #
      # @note   the bridge support metadata is added to the resources of the
      #         library because it is needed for environments interpreted at
      #         runtime.
      #
      # @return [void]
      #
      def create_bridge_support_file
        if target_definition.podfile.generate_bridge_support?
          UI.message "- Generating BridgeSupport metadata at #{UI.path library.bridge_support_path}" do
            generator = Generator::BridgeSupport.new(pods.map do |pod|
              pod.relative_header_files.map { |header| sandbox.root + header }
            end.flatten)
            generator.save_as(library.bridge_support_path)
            copy_resources_script.resources << library.bridge_support_name
          end
        end
      end

      # Creates a script that copies the resources to the bundle of the client
      # target.
      #
      # @todo   This should be replaced by an Xcode copy resources build phase.
      #
      # @return [void]
      #
      def create_copy_resources_script
        UI.message "- Generating copy resources script at #{UI.path(library.copy_resources_script_path)}" do
          copy_resources_script.resources << pods.map { |p| p.relative_resource_files }.flatten
          copy_resources_script.save_as(library.copy_resources_script_path)
        end
      end

      #-----------------------------------------------------------------------#

      # @!group Private helpers.

      private

      # @return [PBXFileReference] the file reference to the xcconfig file of
      #         the target.
      #
      # @note   Generated by the {#add_file_reference_for_support_files} step.
      #
      attr_reader :xcconfig_file_ref


      # @return [Project] the Pods project of the sandbox.
      #
      def project
        sandbox.project
      end

      def pods
        library.local_pods
      end

      # @return [TargetDefinition] the target definition of the library.
      #
      def target_definition
        library.target_definition
      end

      # Creates and caches the copy resource script.
      #
      # @return [CopyResourcesScript]
      #
      def copy_resources_script
        @copy_resources_script ||= Generator::CopyResourcesScript.new
      end

      # Returns the Xcode build settings for a target with the given platform.
      #
      # @param  [Platform] platform
      #         the platform for which the build settings are needed.
      #
      # @return [Hash] the build settings.
      #
      def build_settings_for_platform(platform)
        settings = {}
        settings['ARCHS'] = "armv6 armv7" if platform.requires_legacy_ios_archs?

        if dt = platform.deployment_target
          if platform == :ios
            settings['IPHONEOS_DEPLOYMENT_TARGET'] = dt.to_s
          elsif platform == :osx
            settings['MACOSX_DEPLOYMENT_TARGET'] = dt.to_s
          else
            raise Informative, "Unknown platform #{platform}"
          end
        end
        settings
      end

    end
  end
end

