module Pod
  class Installer
    class PodsProjectGenerator

      # Controller class responsible of creating and configuring the static
      # target target in Pods project. It also creates the support file needed
      # by the target.
      #
      class TargetInstaller

        # @return [Sandbox] sandbox the sandbox where the support files should
        #         be generated.
        #
        attr_reader :sandbox

        # @return [target] The target whose target needs to be generated.
        #
        attr_reader :target

        # @param  [Project] project @see project
        # @param  [target] target @see target
        #
        def initialize(sandbox, target)
          @sandbox = sandbox
          @target = target
        end


        private

        # @!group Installation steps
        #---------------------------------------------------------------------#

        # Adds the target for the target to the Pods project with the
        # appropriate build configurations.
        #
        # @note   The `PODS_HEADERS_SEARCH_PATHS` overrides the xcconfig.
        #
        # @return [void]
        #
        def add_target
          name = target.label
          platform = target.platform.name
          deployment_target = target.platform.deployment_target.to_s
          @native_target = project.new_target(:static_target, name, platform, deployment_target)

          settings = {}
          if target.platform.requires_legacy_ios_archs?
            settings['ARCHS'] = "armv6 armv7"
          end

          @native_target.build_settings('Debug').merge!(settings)
          @native_target.build_settings('Release').merge!(settings)

          target.user_build_configurations.each do |bc_name, type|
            @native_target.add_build_configuration(bc_name, type)
          end

          target.target = @native_target
        end

        # Creates the group that holds the references to the support files
        # generated by this installer.
        #
        # @return [void]
        #
        def create_suport_files_group
          @support_files_group = project.support_files_group.new_group(target.name)
        end

        # Generates a dummy source file for each target so libraries that contain
        # only categories build.
        #
        # @return [void]
        #
        def create_dummy_source
          path = target.dummy_source_path
          UI.message "- Generating dummy source file at #{UI.path(path)}" do
            generator = Generator::DummySource.new(target.label)
            generator.save_as(path)
            file_reference = add_file_to_support_group(path)
            target.target.source_build_phase.add_file_reference(file_reference)
          end
        end

        # @return [PBXNativeTarget] the target generated by the installation
        #         process.
        #
        # @note   Generated by the {#add_target} step.
        #
        attr_reader :native_target


        private

        # @!group Private helpers.
        #---------------------------------------------------------------------#

        # @return [Project] the Pods project of the sandbox.
        #
        def project
          sandbox.project
        end

        # @return [TargetDefinition] the target definition of the target.
        #
        def target_definition
          target.target_definition
        end

        # @return [PBXGroup] the group where the file references to the support
        #         files should be stored.
        #
        attr_reader :support_files_group

        # Adds a reference to the given file in the support group of this target.
        #
        # @param  [Pathname] path
        #         The path of the file to which the reference should be added.
        #
        # @return [PBXFileReference] the file reference of the added file.
        #
        def add_file_to_support_group(path)
          support_files_group.new_file(path)
        end

        #-----------------------------------------------------------------------#

      end
    end
  end
end

