module Pod
  class Installer
    # Context object designed to be used with the HooksManager which describes
    # the context of the installer.
    #
    class PostInstallHooksContext
      # @return [String] The path to the sandbox root (`Pods` directory).
      #
      attr_accessor :sandbox_root

      # @return [Project] The Pods Xcode project.
      #
      attr_accessor :pods_project

      # @return [Sandbox] The Sandbox for the project.
      #
      attr_accessor :sandbox

      # @return [Array<UmbrellaTargetDescription>] The list of
      #         the CocoaPods umbrella targets generated by the installer.
      #
      attr_accessor :umbrella_targets

      # @return [PostInstallHooksContext] Convenience class generator method
      #
      # @param  [Sandbox] sandbox
      #         The sandbox
      #
      # @param  [Array<AggregateTarget>] aggregate_targets
      #         The aggregate targets, which will been presented by an adequate
      #         {UmbrellaTargetDescription} in the generated context.
      #
      # @return [HooksContext] Convenience class method to generate the
      #         static context.
      #
      def self.generate(sandbox, aggregate_targets)
        umbrella_targets_descriptions = []
        aggregate_targets.each do |umbrella|
          desc = UmbrellaTargetDescription.new
          desc.user_project = umbrella.user_project
          desc.user_targets = umbrella.user_targets
          desc.specs = umbrella.specs
          desc.platform_name = umbrella.platform.name
          desc.platform_deployment_target = umbrella.platform.deployment_target.to_s
          desc.cocoapods_target_label = umbrella.label
          umbrella_targets_descriptions << desc
        end

        result = new
        result.sandbox_root = sandbox.root.to_s
        result.pods_project = sandbox.project
        result.sandbox = sandbox
        result.umbrella_targets = umbrella_targets_descriptions
        result
      end

      # Pure data class which describes and umbrella target.
      #
      class UmbrellaTargetDescription
        # @return [Xcodeproj::Project] The user project into which this target
        #         is integrated.
        #
        attr_accessor :user_project

        # @return [String] The path of the user project
        #         integrated by this target.
        #
        def user_project_path
          user_project.path if user_project
        end

        # @return [Array<PBXNativeTarget>]
        #         The list of user targets integrated by this umbrella target.
        #
        attr_accessor :user_targets

        # @return [Array<String>] The list of the UUIDs of the
        #         user targets integrated by this umbrella
        #         target.  They can be used to find the
        #         targets opening the project They can be used
        #         to find the targets opening the project with
        #         Xcodeproj.
        #
        def user_target_uuids
          user_targets.map(&:uuid)
        end

        # @return [Array<Specification>] The list of the
        #         specifications of the target.
        #
        attr_accessor :specs

        # @return [Symbol] The platform (either `:ios`, `:watchos`, `:tvos`, or `:osx`).
        #
        attr_accessor :platform_name

        # @return [String] The deployment target.
        #
        attr_accessor :platform_deployment_target

        # @return [String] The label for the target.
        #
        attr_accessor :cocoapods_target_label
      end
    end
  end
end
