module Pod
  class Command
    class Setup < Command
      self.summary = 'Setup the CocoaPods environment'

      self.description = <<-DESC
        Creates a directory at `~/.cocoapods' which will hold your spec-repos.
        This is where it will create a clone of the public `master' spec-repo from:

            https://github.com/CocoaPods/Specs

        If the clone already exists, it will ensure that it is up-to-date.
      DESC

      def self.options
        [["--push", "Use this option to enable push access once granted"]].concat(super)
      end

      extend Executable
      executable :git

      def initialize(argv)
        @push_option  = argv.flag?('push')
        super
      end

      def self.dir
        Config.instance.repos_dir + 'master'
      end

      def dir
        self.class.dir
      end

      def read_only_url
        'https://github.com/CocoaPods/Specs.git'
      end

      def read_write_url
        'git@github.com:CocoaPods/Specs.git'
      end

      def url
        if push?
          read_write_url
        else
          read_only_url
        end
      end

      def origin_url_read_only?
        read_master_repo_url.chomp == read_only_url
      end

      def origin_url_push?
        read_master_repo_url.chomp == read_write_url
      end

      def push?
        @push_option || (dir.exist? && origin_url_push?)
      end

      def read_master_repo_url
        Dir.chdir(dir) do
          origin_url = git('config --get remote.origin.url')
        end
      end

      def set_master_repo_url
        Dir.chdir(dir) do
          git("remote set-url origin '#{url}'")
        end
      end

      def add_master_repo
        @command ||= Repo::Add.parse(['master', url, 'master']).run
      end

      def update_master_repo
        Repo::Update.run(['master'])
      end

      def set_master_repo_branch
        Dir.chdir(dir) do
          git("checkout master")
        end
      end

      def self.run_if_needed
        self.new(CLAide::ARGV.new([])).run unless
        dir.exist? && Repo.compatible?('master')
      end

      def run
        UI.section "Setting up CocoaPods master repo" do
          if dir.exist?
            set_master_repo_url
            set_master_repo_branch
            update_master_repo
          else
            add_master_repo
          end
          # Mainly so the specs run with submodule repos
          if (dir + '.git/hooks').exist?
            hook = dir + '.git/hooks/pre-commit'
            hook.open('w') { |f| f << "#!/bin/sh\nrake lint" }
            `chmod +x '#{hook}'`
          end
        end
        UI.puts "Setup completed (#{push? ? "push" : "read-only"} access)".green
      end
    end
  end
end
