require File.expand_path('../../spec_helper', __FILE__)

module Pod
  describe Resolver do
    describe "In general" do
      before do
        @podfile = Podfile.new do
          platform :ios
          pod 'BlocksKit', '1.5.2'
        end
        locked_deps = [Dependency.new('BlocksKit', '1.5.2')]
        @resolver = Resolver.new(config.sandbox, @podfile, locked_deps)
      end

      it "returns the sandbox" do
        @resolver.sandbox.should == config.sandbox
      end

      it "returns the podfile" do
        @resolver.podfile.should == @podfile
      end

      it "returns the locked dependencies" do
        @resolver.locked_dependencies.should == [Dependency.new('BlocksKit', '1.5.2')]
      end

      it "allows to specify whether the external sources should be updated against the remote" do
        @resolver.update_external_specs = true
        @resolver.update_external_specs.should.be.true
      end

      it "allows to specify whether the sandbox can be modified and pre-downloads are allowed" do
        @resolver.allow_pre_downloads = false
        @resolver.allow_pre_downloads.should.be.false
      end

      it "allows pre-downloads by default" do
        @resolver.allow_pre_downloads.should.be.true
      end

      #--------------------------------------#

      it "resolves the specification of the podfile" do
        target_definition = @podfile.target_definitions[:default]
        specs = @resolver.resolve[target_definition]
        specs.map(&:to_s).should == [
          "A2DynamicDelegate (2.0.2)",
          "BlocksKit (1.5.2)",
          "libffi (3.0.11)"
        ]
      end

      it "returns the resolved specifications grouped by target definition" do
        @resolver.resolve
        target_definition = @podfile.target_definitions[:default]
        specs = @resolver.specs_by_target[target_definition]
        specs.map(&:to_s).should == [
          "A2DynamicDelegate (2.0.2)",
          "BlocksKit (1.5.2)",
          "libffi (3.0.11)"
        ]
      end

      it "it resolves specifications from external sources" do
        podspec = fixture('integration/Reachability/Reachability.podspec')
        podfile = Podfile.new do
          platform :ios
          pod "Reachability", :podspec => podspec
        end
        resolver = Resolver.new(config.sandbox, podfile)
        resolver.resolve
        specs = resolver.specs_by_target.values.flatten
        specs.map(&:to_s).should == ['Reachability (3.0.0)']
      end
    end

    #-------------------------------------------------------------------------#

    describe "Resolution" do
      before do
        @podfile = Podfile.new do
          platform :ios, '6.0'
          pod 'BlocksKit', '1.5.2'
        end
        @resolver = Resolver.new(config.sandbox, @podfile)
      end

      it "cross resolves dependencies" do
        @podfile = Podfile.new do
          platform :ios, '6.0'
          pod 'BlocksKit', '=  1.0'
          pod 'libffi',  '<  3.0'
        end

        dependant_1_0 = Spec.new do |s|
          s.name         = 'BlocksKit'
          s.version      = '1.0'
          s.platform     = :ios
          s.dependency   'libffi', '1.0'
        end

        depended_1_0 = Spec.new do |s|
          s.name         = 'libffi'
          s.version      = '1.0'
          s.platform     = :ios
        end

        depended_2_0 = Spec.new do |s|
          s.name         = 'libffi'
          s.version      = '2.0'
          s.platform     = :ios
        end

        Source.any_instance.stubs(:versions).with('BlocksKit').returns([Version.new(1.0)])
        Source.any_instance.stubs(:specification).with('BlocksKit', Version.new('1.0')).returns(dependant_1_0)

        Source.any_instance.stubs(:versions).with('libffi').returns([Version.new(1.0), Version.new(2.0)])
        Source.any_instance.stubs(:specification).with('libffi', Version.new('1.0')).returns(depended_1_0)
        Source.any_instance.stubs(:specification).with('libffi', Version.new('2.0')).returns(depended_2_0)

        resolver = Resolver.new(config.sandbox, @podfile)
        specs = resolver.resolve.values.flatten.map(&:to_s).sort
        specs.should == ["BlocksKit (1.0)", "libffi (1.0)"]
      end

      it "holds the context state, such as cached specification sets" do
        @resolver.resolve
        cached_sets = @resolver.send(:cached_sets)
        cached_sets.values.sort_by(&:name).should == [
          SourcesManager.search_by_name('A2DynamicDelegate').first,
          SourcesManager.search_by_name('BlocksKit').first,
          SourcesManager.search_by_name('libffi').first
        ].sort_by(&:name)
      end

      it "raises once any of the dependencies does not match the platform of its podfile target" do
        Specification.any_instance.stubs(:available_platforms).returns([Platform.new(:ios, '999')])
        e = lambda { @resolver.resolve }.should.raise Informative
        e.message.should.match(/platform .* not compatible/)
      end

      it "does not raise if all dependencies are supported by the platform of the target definition" do
        lambda { @resolver.resolve }.should.not.raise
      end

      it "includes all the subspecs of a specification node" do
        @podfile = Podfile.new do
          platform :ios
          pod 'RestKit', '0.10.3'
        end
        resolver = Resolver.new(config.sandbox, @podfile)
        resolver.resolve.values.flatten.map(&:name).sort.should == %w{
        FileMD5Hash
        ISO8601DateFormatter
        JSONKit
        LibComponentLogging-Core
        LibComponentLogging-NSLog
        NSData+Base64
        RestKit
        RestKit/JSON
        RestKit/Network
        RestKit/ObjectMapping/CoreData
        RestKit/ObjectMapping/JSON
        RestKit/UI
        SOCKit
        cocoa-oauth
        }
      end

      it "handles correctly subspecs from external sources" do
        @podfile = Podfile.new do
          platform :ios
          pod 'MainSpec/FirstSubSpec', :git => 'GIT-URL'
        end
        spec = Spec.new do |s|
          s.name         = 'MainSpec'
          s.version      = '1.2.3'
          s.platform     = :ios

          s.subspec 'FirstSubSpec' do |fss|
            fss.source_files = 'some/file'
            fss.subspec 'SecondSubSpec'
          end
        end
        ExternalSources::GitSource.any_instance.stubs(:specification).returns(spec)
        resolver = Resolver.new(config.sandbox, @podfile)
        specs = resolver.resolve.values.flatten.map(&:name).sort
        specs.should == %w{ MainSpec/FirstSubSpec MainSpec/FirstSubSpec/SecondSubSpec }
      end

      it "marks a specification's version to be a HEAD version" do
        podfile = Podfile.new do
          platform :ios
          pod 'FileMD5Hash'
          pod 'JSONKit', :head
        end
        resolver = Resolver.new(config.sandbox, podfile)
        filemd5hash, jsonkit = resolver.resolve.values.first.sort_by(&:name)
        filemd5hash.version.should.not.be.head
        jsonkit.version.should.be.head
      end

      it "raises if it finds two conflicting dependencies" do
        podfile = Podfile.new do
          platform :ios
          pod 'JSONKit', "1.4"
          pod 'JSONKit', "1.5pre"
        end
        resolver = Resolver.new(config.sandbox, podfile)
        e = lambda {resolver.resolve}.should.raise Pod::StandardError
        e.message.should.match(/already activated version/)
      end

      it "takes into account locked dependencies" do
        podfile = Podfile.new do
          platform :ios
          pod 'JSONKit', "<= 1.5pre"
        end
        resolver = Resolver.new(config.sandbox, podfile)
        version = resolver.resolve.values.flatten.first.version
        version.to_s.should == '1.5pre'

        locked_deps = [Dependency.new('JSONKit', "= 1.4")]
        resolver = Resolver.new(config.sandbox, podfile, locked_deps)
        version = resolver.resolve.values.flatten.first.version
        version.to_s.should == '1.4'
      end
    end

    #-------------------------------------------------------------------------#


    describe "#set_from_external_source" do
      before do
        @resolver = Resolver.new(config.sandbox, nil)
      end

      it "it fetches the specification from either the sandbox or from the remote be default" do
        dependency = Dependency.new('Name', :git => 'www.example.com')
        ExternalSources::GitSource.any_instance.expects(:specification_from_external).returns(Specification.new).once
        @resolver.send(:set_from_external_source, dependency)
      end

      it "it fetches the specification from the remote if in update mode" do
        dependency = Dependency.new('Name', :git => 'www.example.com')
        ExternalSources::GitSource.any_instance.expects(:specification).returns(Specification.new).once
        @resolver.update_external_specs = false
        @resolver.send(:set_from_external_source, dependency)
      end

      it "it fetches the specification only from the sandbox if pre-downloads are disabled" do
        dependency = Dependency.new('Name', :git => 'www.example.com')
        Sandbox.any_instance.expects(:specification).returns(Specification.new).once
        @resolver.allow_pre_downloads = true
        @resolver.send(:set_from_external_source, dependency)
      end
    end

    #-------------------------------------------------------------------------#

  end
end
