from django.core import exceptions
from django.db import models
from django.utils.translation import ugettext_lazy as _

from gm_serializer.models.descriptor import WeakDescriptor
from gm_serializer.manager import WeakRelManager

INT = 'INT'
VARCHAR = 'VARCHAR'


class WeakRelField(models.Field):

    default_error_messages = {
        "invalid": _("'%(value)s' is not a valid %(type)")
    }
    relation_class = WeakRelManager
    descriptor_class = WeakDescriptor

    def __init__(self, *args, **kwargs):
        """

        :param args:
        :param kwargs:
        """
        self.url = kwargs.pop('url')
        self.type = kwargs.pop('type', int)
        self.args = kwargs.pop('args', 'id')
        self.batch = kwargs.pop('batch', True)
        self.relation = self.relation_class(
            url=self.url,
            args=self.args,
        )
        super(WeakRelField, self).__init__(*args, **kwargs)

    def get_internal_type(self):
        return "VirtualForeignKey"

    def db_type(self, connection):
        if self.type is int:
            return INT
        elif self.type is str:
            return VARCHAR
        raise RuntimeError("Unsupported data type for VirtualForeignKey: %s" % self.type)

    def get_db_prep_value(self, value, connection, prepared=False):
        """
        lazy foreign key db prep value.
        :param value:
        :param connection:
        :param prepared:
        :return:
        """
        if value is None:
            return None

        if hasattr(value, 'id'):
            return self.type(value.id)

        try:
            return self.type(value)
        except (TypeError, ValueError):
            raise exceptions.ValidationError(
                self.error_messages['invalid'],
                code='invalid',
                params={'value': value, 'type': str(self.type)}
            )

    def contribute_to_class(self, cls, name, **kwargs):
        """
        :param cls:
        :param name:
        :param kwargs:
        :return:
        """
        super(WeakRelField, self).contribute_to_class(cls, name, **kwargs)
        setattr(cls, self.name, self.descriptor_class(self))

    def deconstruct(self):
        """
        :param
        :return:
        """
        name, path, args, kwargs = super(WeakRelField, self).deconstruct()
        kwargs.update({
            'url': self.url, 'type': self.type, 'args': self.args, 'batch': self.batch
        })
        return name, path, args, kwargs

