
from django.db import models
from rest_framework import serializers
from rest_framework.serializers import ListSerializer

from gm_serializer.error import LazyObjNotExist, MultiLazyObjFound

LIST_SERIALIZER_KWARGS = list(serializers.LIST_SERIALIZER_KWARGS)
LIST_SERIALIZER_KWARGS.extend(['lazy', 'lazy_field_ids'])


class GmModelSerializer(serializers.ModelSerializer):

    @classmethod
    def many_init(cls, *args, **kwargs):
        allow_empty = kwargs.pop('allow_empty', None)
        lazy = kwargs.pop('lazy', False)
        child_serializer = cls(*args, **kwargs)
        list_kwargs = {
            'child': child_serializer,
            'lazy': lazy,
            'lazy_fields': cls.get_laze_fields(*args, **kwargs)
        }

        if allow_empty is not None:
            list_kwargs['allow_empty'] = allow_empty

        list_kwargs.update({
            key: value for key, value in kwargs.items()
            if key in LIST_SERIALIZER_KWARGS
        })

        meta = getattr(cls, 'Meta', None)
        list_serializer_class = getattr(meta, 'list_serializer_class', GmListSerializer)
        return list_serializer_class(*args, **list_kwargs)

    @classmethod
    def get_laze_fields(cls, *args, **kwargs):
        lazy_fields = {}
        meta = getattr(cls, 'Meta', None)
        model = getattr(meta, 'model')
        lazy_field_names = getattr(meta, 'lazy_fields', [])
        for field_name in lazy_field_names:
            field = getattr(model, field_name).field
            rel = field.relation_class(
                name=field_name,
                url=field.url,
                args=field.args,
                value_lst=cls.get_field_value_list(field_name, args[0])
            )
            lazy_fields[field_name] = rel.get_queryset
        return lazy_fields

    @classmethod
    def get_field_value_list(cls, field_name, obj_list):
        return [getattr(obj, "%s_id" % field_name) for obj in obj_list]


class GmListSerializer(ListSerializer):

    def __init__(self, *args, **kwargs):
        self.lazy = kwargs.pop('lazy', False)
        self.lazy_fields = kwargs.pop('lazy_fields', None)
        super(GmListSerializer, self).__init__(*args, **kwargs)

    def to_representation(self, data):
        iterable = data.all() if isinstance(data, models.Manager) else data
        if self.lazy:
            iterable = self.lazy_iterable(iterable)
        else:
            iterable = self.unlazy_iterable(iterable)
        return [
            self.child.to_representation(item) for item in iterable
        ]

    def lazy_iterable(self, iterable):
        """

        :param iterable:
        :return:
        """
        for field in self.lazy_fields.keys():
            for obj in iterable:
                setattr(obj, '_%s_lazy_' % field, True)
        return iterable

    def unlazy_iterable(self, iterable):
        """

        :param iterable:
        :return:
        """
        for field, future in self.lazy_fields.items():
            assert callable(future)
            futures = future()
            iterable = self.zip_iterable(field, iterable, futures)
        return iterable

    @classmethod
    def zip_iterable(cls, field, iterable, futures):
        for i in range(len(iterable)):
            lazy_id = getattr(iterable[i], "%s_id" % field)
            future = filter(lambda item: item['id'] == lazy_id, futures)
            if len(future) > 1:
                raise MultiLazyObjFound
            elif len(future) == 1:
                setattr(iterable[i], field, future[0])
            else:
                raise LazyObjNotExist
        return iterable

    def update(self, instance, validated_data):
        pass
