# coding:utf-8

"""首页推荐相关。"""

import json
from datetime import datetime

from libs.cache import redis_client


class DeviceRecommend(object):
    """记录当前推荐信息。"""

    tpl_key = "{tab_type}-{feed_type}-{city_id}-{offset}-{date}"

    @staticmethod
    def current_date():
        return datetime.now().strftime('%Y-%m-%d')

    @classmethod
    def load(cls, tab_type, feed_type, city_id, offset):
        """获取设备的未读的推荐信息ID列表。

        :param feed_type: feed类型：问题、回答...
        :return: 返回已读的id列表
        """

        key = cls.tpl_key.format(city_id=city_id, offset=offset, tab_type=tab_type,
                                 feed_type=feed_type, date=cls.current_date())
        ids = redis_client.get(key)

        try:
            id_list = json.loads(ids.decode("utf-8"))
        except:
            id_list = [int(item) for item in ids.decode("utf-8").split(',')] if ids else []

        return id_list

    @classmethod
    def store(cls, tab_type, feed_type, city_id, offset, id_list, ex=2 * 60 * 60):
        """保存或者更新用户未读
        的推荐信息ID列表

        :param tab_type: ...
        :param feed_type: feed类型：问题、回答...
        :param id_list: 需要更新的推荐信息id列表
        :param city_id: 城市ID
        :param ex: 设置过期时间，默认2小时，没有则当天24：00过期
        :return: 返回已读的id列表
        """

        if not id_list:
            return

        key = cls.tpl_key.format(city_id=city_id, offset=offset, tab_type=tab_type,
                                 feed_type=feed_type, date=cls.current_date())

        if not ex:
            # 当天剩余时间设置为过期时间
            now = datetime.now()
            ex = int((now.replace(hour=0, second=0, minute=0, microsecond=0, day=now.day+1) - now).total_seconds())

        redis_client.set(key, json.dumps(id_list), ex)


class DeviceRecommendRead(object):
    """记录设备已读推荐信息。"""

    tpl_key = "{device_id}-{tab_type}-{feed_type}-{city_id}-{offset}-{date}-read"

    @staticmethod
    def current_date():
        return datetime.now().strftime('%Y-%m-%d')

    @classmethod
    def load(cls, device_id, tab_type, feed_type, city_id, offset):
        """获取设备的已读的推荐信息ID列表。

        :param device_id: 用户设备ID
        :param city_id: 城市ID
        :param feed_type: feed类型：问题、回答...
        :return: 返回已读的id列表
        """

        key = cls.tpl_key.format(device_id=device_id, feed_type=feed_type, tab_type=tab_type,
                                 offset=offset, city_id=city_id, date=cls.current_date())
        ids = redis_client.get(key)

        try:
            id_list = json.loads(ids.decode("utf-8"))
        except:
            id_list = [int(item) for item in ids.decode("utf-8").split(',')] if ids else []

        return id_list

    @classmethod
    def store(cls, device_id, tab_type, feed_type, city_id, offset, id_list, ex=None):
        """保存或者更新用户已读
        的推荐信息ID列表

        :param device_id: 用户设备ID
        :param city_id: 城市ID
        :param feed_type: feed类型：问题、回答...
        :param id_list: 需要更新的推荐信息id列表
        :param ex: 设置过期时间，没有则当天24：00过期
        :return: 返回已读的id列表
        """

        if not id_list:
            return

        key = cls.tpl_key.format(device_id=device_id, feed_type=feed_type, tab_type=tab_type,
                                 offset=offset, city_id=city_id, date=cls.current_date())

        if not ex:
            # 当天剩余时间设置为过期时间
            now = datetime.now()
            ex = int((now.replace(hour=0, second=0, minute=0, microsecond=0, day=now.day+1) - now).total_seconds())

        redis_client.set(key, json.dumps(id_list), ex)
