import os
import shutil
import time
from datetime import datetime
from pathlib import Path

import tensorflow as tf
from sklearn.model_selection import train_test_split

from models.esmm.fe import click_fe, device_fe, fe, tractate_fe
from models.esmm.input_fn import esmm_input_fn


def main():
    time_begin = time.time()

    tf.compat.v1.logging.set_verbosity(tf.compat.v1.logging.INFO)

    data_path = Path("~/data/cvr_data").expanduser()  # local
    # data_path = Path("/srv/apps/node2vec_git/cvr_data/")  # server
    tractate_df, tractate_click_df, tractate_conversion_df = tractate_fe.read_csv_data(data_path)
    tractate_df = tractate_fe.tractate_feature_engineering(tractate_df)
    device_df = device_fe.read_csv_data(data_path)
    device_df = device_fe.device_feature_engineering(device_df)

    cc_df = click_fe.click_feature_engineering(tractate_click_df, tractate_conversion_df)
    df = tractate_fe.join_features(device_df, tractate_df, cc_df)
    # for i in df.columns:
    #     print(i)
    # print(df.dtypes)

    train_df, test_df = train_test_split(df, test_size=0.2)
    train_df, val_df = train_test_split(train_df, test_size=0.2)

    all_features = fe.build_features(df, tractate_fe.INT_COLUMNS, tractate_fe.FLOAT_COLUMNS, tractate_fe.CATEGORICAL_COLUMNS)
    params = {"feature_columns": all_features, "hidden_units": [64, 32], "learning_rate": 0.1}
    model_path = str(Path("~/data/model_tmp/").expanduser())
    # if os.path.exists(model_path):
    #     shutil.rmtree(model_path)

    session_config = tf.compat.v1.ConfigProto()
    session_config.gpu_options.allow_growth = True
    session_config.gpu_options.per_process_gpu_memory_fraction = 0.9
    estimator_config = tf.estimator.RunConfig(session_config=session_config)

    model = tf.estimator.Estimator(model_fn=esmm_model_fn, params=params, model_dir=model_path, config=estimator_config)
    train_spec = tf.estimator.TrainSpec(input_fn=lambda: esmm_input_fn(train_df, shuffle=True), max_steps=50000)
    eval_spec = tf.estimator.EvalSpec(input_fn=lambda: esmm_input_fn(val_df, shuffle=False))
    tf.estimator.train_and_evaluate(model, train_spec, eval_spec)

    model_export_path = str(Path("~/data/models/tractate/").expanduser())
    save_path = model_export(model, all_features, model_export_path)
    print("save to: " + save_path)

    total_time = (time.time() - time_begin) / 60
    print("total cost {:.2f} mins at {}".format(total_time, datetime.now()))


if __name__ == "__main__":
    main()
