# coding:utf8
from django.conf import settings
from django.contrib.auth.models import User
from django.contrib.auth.hashers import make_password
from gm_types.gaia import LOGIN_AUTH_TYPE

from rpc.tool.random_tool import random_str

from api.tool.user_tool import user_register_log_data
from api.tool.common_tool import check_platform
from api.tool.notification_tool import send_notification
from api.models import Person, UserExtra


def create_user_by_email(ctx, email):
    try:
        Person.objects.get(email=email)
        return None
    except Person.DoesNotExist:
        pass

    try:
        user = User.objects.get(username=email)
    except User.DoesNotExist:
        from api.tasks.user_related_tasks import follow_suozhang
        user = UserExtra.create_user(email, "", random_str(32), login_auth_type=LOGIN_AUTH_TYPE.Email)
        follow_suozhang.delay(user.id)
        ctx.logger.app(**user_register_log_data(user_id=user.id))

    return user


def create_user_by_phone(ctx, phone):
    try:
        Person.objects.get(phone=phone)
        return None
    except Person.DoesNotExist:
        pass

    try:
        user = User.objects.get(username=phone)
    except User.DoesNotExist:
        from api.tasks.user_related_tasks import follow_suozhang
        user = UserExtra.create_user(phone, "", random_str(32), login_auth_type=LOGIN_AUTH_TYPE.phone)
        follow_suozhang.delay(user.id)
        try:
            send_notification(
                uid=user.id, title=u'所长欢迎您',
                content=settings.WELCOME_CONTENT,
                url='gengmei://message_chat?user_key=22_{}'.format(user.id)
            )
        except:
            pass
        ctx.logger.app(**user_register_log_data(user_id=user.id))

    return user


def create_person_by_phone(phone, user, platform, password, area_id):
    platform = check_platform(platform)
    person = Person.objects.create(phone=phone, user=user, platform=platform, area_id=area_id)
    if password:
        update_password(person, password)
    user.userextra.phone = phone
    user.userextra.save()
    return person


def create_person_by_email(email, user, platform, password, area_id):
    # 由于医生端鉴权用的是user表,用户端鉴权用的person表,需要双写
    platform = check_platform(platform)
    person = Person.objects.create(email=email, user=user, platform=platform, area_id=area_id)
    update_password(person, password)
    return person


def update_password(person, raw_password=None, hash_password=None):
    """更新密码，同时更新Person, User里面的password, 并保持一致
    """
    if raw_password is None and hash_password is None:
        raise TypeError('expected at least 2 arguments')
    if raw_password is not None:
        hash_password = generate_password_hash(raw_password)
    Person.objects.filter(id=person.id).update(password=hash_password)
    person.password = hash_password
    User.objects.filter(id=person.user.id).update(password=hash_password)
    person.user.password = hash_password


def generate_password_hash(password):
    """生成密码hash
    """
    return make_password(password, None, 'pbkdf2_sha256')
