# -*- coding: utf-8 -*-
from gm_types.error import ERROR

from api.services.beauty_service import BeautyService
from api.services.word_service import WordService
from api.tool.user_tool import get_user_from_context
from rpc.decorators import bind_context, bind
from rpc.tool.error_code import gen

from api.services.user_face_services import (
    FaceService,
    FacialService,
    FacialFeatureService
)

from api.services.tag_service import list_tags_by_polymer_ids


@bind_context('api/user/face_image')
def get_user_face_image(ctx, face_id=None):
    """
    获取用户扫脸图片
    :param ctx:
    :param face_id:
    :return:
    """
    user = get_user_from_context(ctx)
    if not user:
        return gen(ERROR.LOGIN_REQUIRED)

    ret = FaceService.get_one_image(user_id=user.id, face_id=face_id)
    if not ret:
        return gen(ERROR.FACE_IMAGE_NOT_FOUND)

    return ret


@bind_context('api/face_image/add')
def add_one_image(ctx, face_id, face_url, beauty_url):
    user = get_user_from_context(ctx)
    if not user:
        return gen(ERROR.LOGIN_REQUIRED)

    res = {"face_id": face_id}

    ret = FaceService.create_one_image(face_id=face_id, user_id=user.id, face_url=face_url)
    if not ret:
        return gen(ERROR.FACE_IMAGE_CREATE_FAILURE)

    data = BeautyService.get_beauty_info(image_url=beauty_url)
    if not data:
        return res

    beauty_photo_id = data.get('id')

    if not beauty_photo_id:
        return res

    word_parent_info = WordService.get_word_by_beauty_photo(beauty_photo_id=beauty_photo_id)

    data['word_parent_info'] = word_parent_info
    res.update(data)

    return res


@bind_context('api/user_face/share')
def beauty_info(ctx, image_url):
    """获取美图分享信息"""

    user = get_user_from_context(ctx)
    if not user:
        return gen(ERROR.LOGIN_REQUIRED)

    data = BeautyService.get_beauty_info(image_url=image_url)

    word_parent_info = WordService.get_word_by_beauty_photo(url=image_url)

    data['word_parent_info'] = word_parent_info

    return data


@bind("api/one_image/create")
def create_one_image_info(face_id, face_url="", device_id="", user_id=0):
    """
    新版记录用户的扫脸数据
    :param face_id:
    :param face_url:
    :param device_id:
    :param user_id:
    :return:
    """
    return FaceService.create_one_image_v2(
        face_id=face_id,
        face_url=face_url,
        device_id=device_id,
        user_id=user_id
    )


@bind("api/one_image/get_by_face_id")
def get_face_url_by_face_id(face_id):
    """
    通过face_id 获取 对应扫脸图片
    :param face_id:
    :return:
    """
    return FaceService.get_one_image_v2(face_id)


@bind("api/one_image/get_facial_feature")
def get_facial_feature():
    """
    获取五官分类信息
    :return:
    """
    _infos = FacialFeatureService.get_online_info()
    polymer_ids = [int(item.get('polymer_id', 0)) for item in _infos]
    polymer_tags = list_tags_by_polymer_ids(polymer_ids)
    for info in _infos:
        info['tags'] = polymer_tags.get(int(info.get('polymer_id')))

    return _infos


@bind("api/one_image/facial_display")
def get_facial_display(contour, chin, eyebrow, eye, nose, lip):
    """
    获取面部部位对应的文案
    :param contour:
    :param chin:
    :param eyebrow:
    :param eye:
    :param nose:
    :param lip:
    :return:
    """

    return FacialService.get_desc_by_classification(
        contour=contour,
        chin=chin,
        eyebrow=eyebrow,
        eye=eye,
        nose=nose,
        lip=lip
    )


@bind("api/one_image/get_face_shape")
def get_face_shape(filters):
    """
    通过筛选，获取脸型及文案
    :param filters: {"contour":"yuanlian", "chin":"jianxiaba"}  筛选条件
    :return:
    """
    return FacialService.get_face_name_and_desc(filters=filters)
