# coding:utf8

from django.db import models
from django.utils import timezone

from gm_types.gaia import (
    CAMPAIGN_MODULE,
    COMPAIGN_STATUS,
    SLIDE_LAYOUT,
    SLIDE_TYPE,
)
from gm_upload import ImgUrlField, IMG_TYPE

from api.tool.datetime_tool import get_timestamp
from api.models import CouponGift
from api.models import Tag, Special, Service, Coupon
from api.models.share import Share


class Campaign(models.Model):
    class Meta:
        app_label = 'api'

    name = models.CharField(verbose_name=u'活动名称', max_length=20)
    online_time = models.DateTimeField(null=True, blank=True, verbose_name=u"上线时间")
    banner = ImgUrlField(img_type=IMG_TYPE.BANNER, max_length=120, verbose_name='banner', blank=True, default='')
    title = models.CharField(max_length=128, blank=True, default='', verbose_name='主标题')
    subtitle = models.CharField(max_length=128, blank=True, default='', verbose_name='副标题')
    rule = models.TextField(null=True, blank=True, verbose_name=u'活动规则')
    offline_time = models.DateTimeField(null=True, blank=True, verbose_name=u'下线时间')
    status = models.IntegerField(verbose_name=u'活动状态', choices=COMPAIGN_STATUS, default=COMPAIGN_STATUS.PENDING)
    is_coupons = models.BooleanField(default=False, help_text=u"是否优惠券", verbose_name=u"是否优惠券")
    is_seckill = models.BooleanField(default=False, help_text=u"秒杀", verbose_name=u"秒杀")
    is_special = models.BooleanField(default=False, help_text=u"专题", verbose_name=u"专题")
    is_overflow = models.BooleanField(default=False, help_text=u"超值", verbose_name=u"超值")
    is_online = models.BooleanField(default=False, help_text=u"是否上线", verbose_name=u"是否上线")
    share = models.ForeignKey(Share, default=None, null=True)

    def data(self):
        return {
            "name": self.name,
            "online_time": get_timestamp(self.online_time),
            "banner": self.banner,
            "title": self.title,
            "subtitle": self.subtitle,
            "rule": self.rule,
            "offline_time": get_timestamp(self.offline_time),
            "status": self.status,
            "is_coupons": self.is_coupons,
            "is_seckill": self.is_seckill,
            "is_special": self.is_special,
            "is_overflow": self.is_overflow,
            'is_online': self.is_online,
            'share': self.share and self.share.to_dict() or None,
        }


class CampaignLayout(models.Model):
    class Meta:
        verbose_name = u'活动布局'
        app_label = 'api'

    campaign = models.ForeignKey('Campaign')
    module = models.IntegerField(u'活动模块类型', choices=CAMPAIGN_MODULE, default='')
    related = models.TextField(u'关联内容')
    ordering = models.IntegerField(u'排序', default=0, help_text=u'越小越排前')
    is_visible = models.BooleanField(u'是否显示', default=False)


class CampaignImageLink(models.Model):
    class Meta:
        verbose_name = u'活动图片链接'
        verbose_name_plural = u'活动图片链接'
        app_label = 'api'

    name = models.CharField(u'图片链接名', max_length=50, default='')
    link_type = models.CharField(
        u'跳转的类型', max_length=2,
        choices=SLIDE_TYPE, default=SLIDE_TYPE.TOPIC,
        db_index=True,
    )
    image_layout = models.CharField(
        u'图片布局', max_length=2,
        choices=SLIDE_LAYOUT, default=SLIDE_LAYOUT.CAROUSEL,
        db_index=True,
    )
    image = ImgUrlField(u'图片地址', img_type=IMG_TYPE.BANNER, max_length=1024, null=False)
    image1 = ImgUrlField(u'图片地址256*640', img_type=IMG_TYPE.BANNER, max_length=1024, null=True, default='')
    topic_id = models.IntegerField(verbose_name=u'指定的话题', null=True, blank=True)
    service = models.ForeignKey(Service, verbose_name=u'指定的服务', null=True, blank=True)
    diary_id = models.IntegerField(verbose_name=u'指定的日记', null=True, blank=True)
    tag = models.ForeignKey(Tag, verbose_name=u'指定的Tag列表', null=True, blank=True)
    activity_id = models.IntegerField(verbose_name=u'免费活动', null=True, blank=True)
    special = models.ForeignKey(Special, verbose_name=u'指定的福利专题', null=True, blank=True)
    url = models.CharField(max_length=1024, null=True, blank=True, verbose_name=u'跳转的网址')
    ordering = models.IntegerField(default=20, verbose_name=u"展示顺序", help_text=u"小的排在前，大的排在后")
    created_time = models.DateTimeField(auto_now_add=True, verbose_name=u'添加时间')
    effect_time = models.DateTimeField(null=True, verbose_name=u'生效时间', default=timezone.now)
    end_time = models.DateTimeField(null=True, verbose_name=u'下线时间', blank=True)
    is_online = models.BooleanField(u'是否上线', default=True, help_text=u"是否上线")
    module_type = models.IntegerField(
        u'模块类型', choices=CAMPAIGN_MODULE, default=CAMPAIGN_MODULE.IMGMIX,
        help_text=u'该图片链接所属的模块类型',
    )


class CampaignCouponPage(models.Model):
    class Meta:
        verbose_name = u'活动美券页配置'
        app_label = 'api'

    campaign = models.OneToOneField('Campaign')
    banner = ImgUrlField(u'图片地址', img_type=IMG_TYPE.BANNER, max_length=1024, default='')
    coupons = models.ManyToManyField(Coupon, through='CampaignCoupon', related_name='campaigns')
    gifts = models.ManyToManyField(CouponGift, through='CampaignGift', related_name='giftcampaigns')
    rule = models.TextField(u'规则说明')
    share = models.ForeignKey(Share)


class CampaignGift(models.Model):
    class Meta:
        verbose_name = u'活动大礼包关联'
        app_label = 'api'

    campaigncouponpage = models.ForeignKey('CampaignCouponPage')
    gift = models.ForeignKey(CouponGift)
    ordering = models.IntegerField(u'展示顺序', default=20, help_text=u'小的排在前，大的排在后')


class CampaignCoupon(models.Model):
    class Meta:
        verbose_name = u'活动美券关联'
        app_label = 'api'

    campaigncouponpage = models.ForeignKey('CampaignCouponPage')
    coupon = models.ForeignKey(Coupon)
    ordering = models.IntegerField(u'展示顺序', default=20, help_text=u'小的排在前，大的排在后')
