# coding=utf8
from __future__ import unicode_literals, absolute_import, print_function
from collections import defaultdict

from django.contrib.auth.models import User
from django.db import models
from django.utils import timezone
from gm_types.gaia import XIAOYING_GENDER
from gm_upload import ImgUrlField, IMG_TYPE

from gm_types.gaia import LOGIN_AUTH_TYPE, FREER_RECRUITMENT_PUSH_TYPE
from api.models.service import Service
from hippo.models import Doctor
from talos import Diary


class FreeRecruitment(models.Model):
    class Meta:
        verbose_name = u'免费招募活动'
        db_table = 'api_freerecruitment'
        app_label = 'api'

    title = models.CharField(max_length=128, null=False, verbose_name=u'活动名称')
    deposit = models.IntegerField(verbose_name=u'押金', null=True, blank=True)
    quota = models.IntegerField(verbose_name=u'名额', default=0)
    start_time = models.DateTimeField(verbose_name=u'起始时间', default=timezone.now)
    end_time = models.DateTimeField(verbose_name=u'结束时间', null=True, blank=True)
    service = models.ForeignKey(Service, verbose_name=u'关联的美购', null=True, blank=True)
    list_image = ImgUrlField(img_type=IMG_TYPE.NOWATERMARK, max_length=128, null=False, verbose_name='列表banner')
    recruitment_rules = models.TextField(default='', blank=True, max_length=2000, verbose_name=u'招募规则')
    diary_requirements = models.TextField(default='', blank=True, max_length=2000, verbose_name=u'日记要求')
    careful_matter = models.TextField(default='', blank=True, max_length=2000, verbose_name=u'注意事项')
    is_synchronize_app = models.BooleanField(u"是否同步到App", null=False, default=False)
    is_online = models.BooleanField(default=True, verbose_name=u'是否上线')

    create_time = models.DateTimeField(verbose_name=u'创建时间', auto_now_add=True)
    update_time = models.DateTimeField(verbose_name=u'更新时间', auto_now=True, db_index=True)


class FreeRecruitmentImage(models.Model):
    class Meta:
        verbose_name = u'免费招募活动详情页头图'
        db_table = 'api_freerecruitment_image'
        app_label = 'api'

    freerecruitment = models.ForeignKey(FreeRecruitment, verbose_name=u'免费招募', related_name='freerecruitmentimage')
    image = ImgUrlField(img_type=IMG_TYPE.NOWATERMARK, max_length=128, null=False, verbose_name='图片')

    create_time = models.DateTimeField(verbose_name=u'创建时间', auto_now_add=True)
    update_time = models.DateTimeField(verbose_name=u'更新时间', auto_now=True, db_index=True)


class FreeRecruitmentDiary(models.Model):
    class Meta:
        verbose_name = u'免费招募活动关联日记'
        db_table = 'api_freerecruitment_diary'
        app_label = 'api'

    freerecruitment = models.ForeignKey(FreeRecruitment, verbose_name=u'免费招募', related_name='freerecruitmentdiary')
    diary = models.ForeignKey(Diary, verbose_name=u"关联的日记本", null=True, blank=True, default=None)

    create_time = models.DateTimeField(verbose_name=u'创建时间', auto_now_add=True)
    update_time = models.DateTimeField(verbose_name=u'更新时间', auto_now=True, db_index=True)

class FreeRecruitmentDoctor(models.Model):
    class Meta:
        verbose_name = u'免费招募活动关联医生'
        db_table = 'api_freerecruitment_doctor'
        app_label = 'api'

    freerecruitment = models.ForeignKey(FreeRecruitment, verbose_name=u'免费招募', related_name='freerecruitmentdoctor')
    doctor = models.ForeignKey(Doctor, verbose_name=u"关联的医生", null=True, blank=True, default=None)

    create_time = models.DateTimeField(verbose_name=u'创建时间', auto_now_add=True)
    update_time = models.DateTimeField(verbose_name=u'更新时间', auto_now=True, db_index=True)


class FreeRecruitmentUser(models.Model):
    class Meta:
        verbose_name = u'免费招募报名用户'
        db_table = 'api_freerecruitmentuser'
        app_label = 'api'

    freerecruitment = models.ForeignKey(FreeRecruitment, verbose_name=u'免费招募', related_name='freerecruitment')
    user = models.ForeignKey(User, verbose_name=u"用户ID")
    name = models.CharField(max_length=128, null=False, verbose_name=u'姓名')
    sex = models.CharField(max_length=2, verbose_name=u'性别', choices=XIAOYING_GENDER)
    age = models.IntegerField(verbose_name=u'年龄', db_index=True)
    user_phone = models.CharField(default='', max_length=20, verbose_name=u'常用手机号')
    full_face_photo = ImgUrlField(img_type=IMG_TYPE.NOWATERMARK, max_length=128, null=True, blank=True, verbose_name='正面照')
    left_45_degrees = ImgUrlField(img_type=IMG_TYPE.NOWATERMARK, max_length=128, null=True, blank=True, verbose_name='左45度照')
    right_45_degrees = ImgUrlField(img_type=IMG_TYPE.NOWATERMARK, max_length=128, null=True, blank=True, verbose_name='右45度照')
    left_image = ImgUrlField(img_type=IMG_TYPE.NOWATERMARK, max_length=128, null=True, blank=True, verbose_name='左侧照')
    right_image = ImgUrlField(img_type=IMG_TYPE.NOWATERMARK, max_length=128, null=True, blank=True, verbose_name='右侧照')
    has_cosmetic_history = models.BooleanField(u"有无整容史", null=False, default=False)
    has_medical_history = models.BooleanField(u"有无既往病史", null=False, default=False)

    create_time = models.DateTimeField(verbose_name=u'创建时间', auto_now_add=True)
    update_time = models.DateTimeField(verbose_name=u'更新时间', auto_now=True, db_index=True)


class AssistanceUser(models.Model):
    class Meta:
        verbose_name = u'免费招募助力用户'
        db_table = 'api_assistanceuser'
        app_label = 'api'

    freerecruitment = models.ForeignKey(FreeRecruitment, verbose_name=u'免费招募')
    user = models.ForeignKey(FreeRecruitmentUser, verbose_name=u"报名用户ID")
    assister = models.ForeignKey(User, verbose_name=u"助力用户ID")

    create_time = models.DateTimeField(verbose_name=u'创建时间', auto_now_add=True)
    update_time = models.DateTimeField(verbose_name=u'更新时间', auto_now=True, db_index=True)

class ReservationUser(models.Model):
    class Meta:
        verbose_name = u'免费招募预约用户'
        db_table = 'api_reservationuser'
        app_label = 'api'

    freerecruitment = models.ForeignKey(FreeRecruitment, verbose_name=u'免费招募')
    user = models.ForeignKey(User, verbose_name=u"预约报名用户ID")

    create_time = models.DateTimeField(verbose_name=u'创建时间', auto_now_add=True)
    update_time = models.DateTimeField(verbose_name=u'更新时间', auto_now=True, db_index=True)


class ThirdAccountInfo(models.Model):

    class Meta:
        verbose_name = "三方账号信息"
        verbose_name_plural = "三方账号信息"
        db_table = 'api_third_account_info'
        app_label = 'api'

    user_id = models.BigIntegerField(verbose_name='用户ID', db_index=True)
    activity_id = models.BigIntegerField(verbose_name='活动id', db_index=True)
    push_type = models.SmallIntegerField(verbose_name='使用push类型', choices=FREER_RECRUITMENT_PUSH_TYPE, default=FREER_RECRUITMENT_PUSH_TYPE.VOTED)
    platform_type = models.SmallIntegerField(verbose_name='平台类型', choices=LOGIN_AUTH_TYPE, default=LOGIN_AUTH_TYPE.Wechat)
    form_id = models.CharField(max_length=180, verbose_name='form_id', null=False, db_index=True)

    is_online = models.BooleanField(default=True, verbose_name=u'是否上线')
    create_time = models.DateTimeField(verbose_name=u'创建时间', auto_now_add=True)
    update_time = models.DateTimeField(verbose_name=u'更新时间', auto_now=True, db_index=True)

    @classmethod
    def upsert(cls, push_type, activity_id, user_id, form_id):

        user, created = cls.objects.update_or_create(
            push_type=push_type,
            activity_id=activity_id,
            user_id=user_id,
            defaults={
                'form_id': form_id,
                'is_online': True
            }
        )

        return user

    @classmethod
    def list(cls, push_type, user_ids):

        ret = defaultdict(dict)
        for user in cls.objects.filter(push_type=push_type, user_id__in=user_ids, is_online=True):
            ret[user.user_id][user.activity_id] = user.form_id

        return ret