#! /usr/bin/env python
# -*- coding: utf-8 -*-

from rpc.decorators import bind, bind_context
from api.models.face.face_puzzle import (
    FacePuzzle,
    FacePuzzleShare,
    FaceChange,
    PuzzleImage
)
from api.tool.user_tool import get_user_from_context
from django.db.models import Q


@bind_context('api/face_app/get_shared_list', login_required=True)
def get_shared_list(ctx, offset=0, size=10):
    """
    获取用户分享过的合照列表
    :param ctx:
    :param count: 获取用户分享信息的个数
    :return:
    """

    user = get_user_from_context(ctx)
    face_list = FacePuzzleShare.objects.filter(sharer_face__user_id=user.id).order_by("-id")[offset:offset+size]

    result = []
    for face_info in face_list:
        result.append(
            {
                "share_id": face_info.id,
                "place": face_info.place,
                "puzzle_type": face_info.puzzle_type,
                "created_time": face_info.created_time,
                "photo_url": face_info.sharer_face.photo_url,
                "yong_photo": face_info.sharer_face.yong_photo,
                "old_photo": face_info.sharer_face.old_photo,
                "laugh_photo": face_info.sharer_face.laugh_photo,
            }
        )
    return result


@bind('api/face_app/share')
def face_app_share(face_id, puzzle_type, sharer_name, place):
    """
    用户分享半张拼图
    :param ctx:
    :param face_id:
    :param puzzle_type:
    :return:
    """
    obj = FacePuzzleShare.objects.create(
        sharer_id=0,
        place=place,
        sharer_face_id=face_id,
        puzzle_type=puzzle_type,
        sharer_name=sharer_name,
    )

    return {"share_id": obj.id}


@bind('api/face_app/puzzle')
def face_app_puzzle_complete(puzzle_share_id, sharer_face_id, participant_face_id, puzzle_type, puzzle_url, participant_name):
    """
    用户完成拼图
    :param ctx:
    :param share_id:
    :param sharer_face_id:
    :param participant_face_id:
    :param puzzle_type:
    :param puzzle_url:
    :return:
    """
    obj = FacePuzzle.objects.create(
        puzzle_share_id=puzzle_share_id,
        sharer_face_id=sharer_face_id,
        participant_face_id=participant_face_id,
        puzzle_type=puzzle_type,
        puzzle_url=puzzle_url,
        participant_name = participant_name,
    )
    return {"share_id": obj.id}


@bind('api/face_app/get_share_info')
def get_share_info(share_id):
    share_obj = FacePuzzleShare.objects.get(id=share_id)
    result = {
        "sharer_face": {
            "user_id": share_obj.sharer_face.user_id,
            "place": share_obj.place,
            "photo_url": share_obj.sharer_face.photo_url,
            "yong_photo": share_obj.sharer_face.yong_photo,
            "old_photo": share_obj.sharer_face.old_photo,
            "laugh_photo": share_obj.sharer_face.laugh_photo,
        },
        "sharer_name": share_obj.sharer_name,
        "puzzle_type": share_obj.puzzle_type
    }

    return result


@bind('api/face_app/get_puzzle_info')
def get_puzzle_info(puzzle_id):

    puzzle_obj = FacePuzzle.objects.get(id=puzzle_id)
    result = {
        "sharer_face": {
            "user_id": puzzle_obj.sharer_face.user_id,
            "photo_url": puzzle_obj.sharer_face.photo_url,
            "yong_photo": puzzle_obj.sharer_face.yong_photo,
            "old_photo": puzzle_obj.sharer_face.old_photo,
            "laugh_photo": puzzle_obj.sharer_face.laugh_photo,
        },
        "participant_face": {
            "user_id": puzzle_obj.participant_face.user_id,
            "photo_url": puzzle_obj.participant_face.photo_url,
            "yong_photo": puzzle_obj.participant_face.yong_photo,
            "old_photo": puzzle_obj.participant_face.old_photo,
            "laugh_photo": puzzle_obj.participant_face.laugh_photo,
        },
        "puzzle_type": puzzle_obj.puzzle_type,
        "puzzle_url": puzzle_obj.puzzle_url,
        "created_time": puzzle_obj.created_time
    }

    return result


@bind_context('api/face_app/get_puzzle_history', login_required=True)
def get_puzzle_info(ctx):
    """
    获取用户是否有过分享
    :param ctx:
    :return: 1 有过分享记录，0 没有分享记录
    """
    user = get_user_from_context(ctx)
    share_obj = FacePuzzleShare.objects.filter(sharer_id=user.id).first()
    if share_obj:
        return {"shared": 1}

    puzzle_query = Q(sharer_face__uer_id=user.id) | Q(participant_face__user_id=user.id)
    puzzle_obj = FacePuzzle.objects.filter(puzzle_query).first()
    if puzzle_obj:
        return {"shared": 1}

    return {"shared": 0}


@bind('api/face_app/save')
def face_app_save_change_info(photo_url, change_type, changed_url):

    """
    每次变脸保存相关信息
    :param ctx:
    :param photo_url:
    :param change_type:
    :param changed_url:
    :return:
    """

    obj, _ = FaceChange.objects.get_or_create(
        user_id=0,
        photo_url=photo_url
    )

    if change_type == 1:
        obj.yong_photo = changed_url
    elif change_type == 10:
        obj.old_photo = changed_url

    obj.save()
    return {"face_id": obj.id}


@bind_context("api/face_app/puzzle_image", login_required=False)
def face_app_puzzle_image(ctx, url_1, url_2, change_type, bg_id, generate_url):
    user = get_user_from_context(ctx)
    user_id = user.id if user else 0
    obj = PuzzleImage.objects.create(
        user_id=user_id,
        img_url_1=url_1,
        img_url_2=url_2,
        change_type=change_type,
        background_id=bg_id,
        generate_url=generate_url,
    )
    return {"generate_id": obj.id}
