# coding=utf-8

from django.db import models
from django.utils import timezone

from api.models import Reservation, Order, RefundOrder, Service
from api.tool.log_tool import logging_exception
from api.tool.datetime_tool import get_timestamp
from gm_types.gaia import TYPES
from gm_types.doctor import NOTIFICATION_TYPE
from services.talos_service import topic_info_for_doctor_notification

from rpc.tool.dict_mixin import to_dict


class Notification(models.Model):
    class Meta:
        verbose_name = u'通知'
        verbose_name_plural = u'通知'
        app_label = 'doctor'
        index_together = [
            ['doctor_id', 'ntype', 'person_id', 'related_item_id', ],
        ]

    doctor_id = models.CharField(max_length=64, verbose_name=u'医生id')
    summary = models.CharField(max_length=100, verbose_name=u'通知内容摘要')
    ntype = models.CharField(max_length=2, verbose_name=u'通知类型', choices=NOTIFICATION_TYPE)

    person_id = models.CharField(max_length=64, verbose_name=u'相关用户（患者）')

    related_item_id = models.CharField(max_length=64, verbose_name=u'关联对象的id')
    related_item_type = models.CharField(max_length=2, verbose_name=u'关联对象的类型', choices=TYPES)

    # 1. 关联预约 保存当时预约的状态reservation.status;
    # 2. 关联订单 保存当时订单的状态order.status
    # 3. 关联退款单 保存当时退款的状态refundorder.status
    # 4. 关联案例 保存更新帖子的topic.id
    related_item_sub = models.CharField(max_length=64, null=True, verbose_name=u'关联对象的的额外信息')

    create_time = models.DateTimeField(verbose_name=u'通知时间', default=timezone.now)
    is_viewed = models.BooleanField(verbose_name=u'用户是否已读', default=False)

    def to_dict(self):
        data = to_dict(self)
        return data

    @property
    def detail_info(self):
        info = {
                'id': self.id,
                'summary': self.summary,
                'ntype': self.ntype,
                'is_viewed': self.is_viewed,
                'create_time': get_timestamp(self.create_time),
                'related_item_type': self.related_item_type,
                'related_item_sub': self.related_item_sub,
                }
        try:
            if self.related_item_type == TYPES.RESERVATION:  # 预约
                reservation = Reservation.objects.get(id=self.related_item_id)
                info['reservation'] = {
                        'id': reservation.id,
                        'date': get_timestamp(reservation.date),
                        'address': reservation.schedule.address.desc,
                        'type': reservation.reservation_type,
                        'service_image': reservation.order.service.image_header,
                        }
            elif self.related_item_type == TYPES.ORDER:  # 订单
                order = Order.objects.get(id=self.related_item_id)
                info['order'] = order.data()
            elif self.related_item_type == TYPES.REFUND_ORDER:  # 退款单
                refund = RefundOrder.objects.get(id=self.related_item_id)
                info['refund'] = refund.data
            elif self.related_item_type == TYPES.DIARY:  # 案例相关
                diary = topic_info_for_doctor_notification(topic_id=self.related_item_sub)
                service_id = diary.pop('service_id')
                diary_title = diary.pop('diary_title')
                if service_id is not None:
                    s = Service.objects.get(id=service_id)
                    title = s.name
                else:
                    title = diary_title
                diary['title'] = title
                info['diary'] = diary
        except:
            logging_exception()
        return info
