#!/usr/bin/env python
# -*- coding:utf-8 -*-
#
#   Author  :   RobertDing
#   E-mail  :   robertdingx@gmail.com
#   Date    :   16/07/04 19:07:06
#   Desc    :   分期model
#
from django.db import models
from django.conf import settings

from gm_crypto.fields import EncryptedCharField
from gm_upload import IMG_TYPE, ImgUrlField, get_private_url_for_long

from api.models import Person, Settlement


class Borrower(models.Model):
    class Meta:
        verbose_name = u'借款人'
        db_table = 'pay_borrower'
        app_label = 'pay'

    partner = models.IntegerField(default=settings.PARTNER_XY)
    person = models.ForeignKey(Person)
    third_uid = models.CharField(max_length=64, verbose_name=u'第三方UID')
    third_password = EncryptedCharField(u'用户第三方密码md5', max_length=64)
    name = EncryptedCharField(u'真实姓名', max_length=16)
    card_id = EncryptedCharField(u'身份证号', max_length=32)
    phone = EncryptedCharField(u"电话", max_length=20)
    gender = models.CharField(max_length=3, default="", verbose_name=u"性别, 中文")
    age = models.IntegerField(default=0)
    marriage = models.CharField(max_length=10, default="", verbose_name=u"婚姻状况")
    education = models.CharField(max_length=10, default="", verbose_name=u"学历")
    province = models.CharField(max_length=20, default="", verbose_name=u"省份")
    city = models.CharField(max_length=20, default="", verbose_name=u"城市")
    district = models.CharField(max_length=40, default="", verbose_name=u"地区")
    address = models.CharField(max_length=64, default="", verbose_name=u"地址")
    created_time = models.DateTimeField(auto_now_add=True)


class Bank(models.Model):
    class Meta:
        verbose_name = u'银行'
        db_table = 'pay_bank'
        app_label = 'pay'

    partner = models.IntegerField(default=settings.PARTNER_XY)
    name = models.CharField(max_length=32, verbose_name=u'银行名称')
    phone = models.CharField(max_length=16, verbose_name=u"银行电话")
    code = models.CharField(max_length=10, verbose_name=u"银行简码")
    logo = ImgUrlField(img_type=IMG_TYPE.BANK, max_length=400, default="")
    description = models.CharField(max_length=32)
    created_time = models.DateTimeField(auto_now_add=True)


class BorrowerBind(models.Model):
    class Meta:
        verbose_name = u'借款人绑定银行'
        db_table = 'pay_borrower_bind'
        app_label = 'pay'

    partner = models.IntegerField(default=settings.PARTNER_XY)
    bank = models.ForeignKey(Bank)
    borrower = models.ForeignKey(Borrower, related_name='binds')
    name = EncryptedCharField(u'持卡人姓名', max_length=16)
    phone = EncryptedCharField(u"银行预留手机换号码", max_length=20)
    card_code = EncryptedCharField(u'银行卡号', max_length=20)
    province = models.CharField(max_length=32, verbose_name=u"开户省份")
    city = models.CharField(max_length=32, verbose_name=u"开户城市")
    sms_code = models.CharField(max_length=10, default="", verbose_name='短信验证码')
    ticket = models.CharField(max_length=64, default="", verbose_name=u"验证凭证")
    created_time = models.DateTimeField(auto_now_add=True)


class BorrowerImage(models.Model):
    class Meta:
        verbose_name = u'借款人证件照'
        db_table = 'pay_image'
        app_label = 'pay'

    partner = models.IntegerField(default=settings.PARTNER_XY)
    borrower = models.ForeignKey(Borrower, related_name='images')
    image_type = models.IntegerField(verbose_name='照片类型')
    url = ImgUrlField(img_type=IMG_TYPE.INSTALLMENT, max_length=400, default="")
    created_time = models.DateTimeField(auto_now_add=True)
    deleted = models.BooleanField(default=False)

    @property
    def public_url(self):
        seconds = settings.IMAGE_FAILURE_SECONDS
        return get_private_url_for_long(self.url, seconds)


class BorrowerRelative(models.Model):
    class Meta:
        verbose_name = u'借款人联系人'
        db_table = 'pay_relative'
        app_label = 'pay'

    partner = models.IntegerField(default=settings.PARTNER_XY)
    borrower = models.ForeignKey(Borrower, related_name='relatives')
    created_time = models.DateTimeField(auto_now_add=True)
    deleted = models.BooleanField(default=False)
    name = EncryptedCharField(u'真实姓名', max_length=16)
    phone = EncryptedCharField(u"电话", max_length=20)
    relationship = models.CharField(max_length=10, verbose_name='关系')
    relation_type = models.IntegerField()


class Installment(models.Model):
    class Meta:
        verbose_name = u'借款单'
        db_table = 'pay_installment'
        app_label = 'pay'

    partner = models.IntegerField(default=settings.PARTNER_XY)
    settlement = models.OneToOneField(Settlement)
    borrower = models.ForeignKey(Borrower, related_name='installments')
    bind_bank = models.ForeignKey(BorrowerBind)
    periods = models.IntegerField(verbose_name=u"期数")
    period_repay = models.IntegerField(default=0, verbose_name=u"每期应还")
    third_loan_id = models.CharField(max_length=64, verbose_name='贷款协议号')
    third_loan_order_id = models.CharField(max_length=64, verbose_name='贷款订单号')
    status = models.IntegerField(verbose_name=u"借款状态", default=0)
    first_repay_time = models.DateTimeField(null=True, verbose_name=u"首次还款时间")
    confirm_time = models.DateTimeField(null=True, verbose_name=u"放款时间")
    refund_time = models.DateTimeField(null=True, verbose_name=u"退款或放弃贷款时间")
    audit_time = models.DateTimeField(null=True, verbose_name=u"审核状态变更时间")
    audit_description = models.CharField(max_length=128, default="", verbose_name=u"对审核结果的简要描述")
    apply_time = models.DateTimeField(auto_now_add=True, verbose_name=u"申请时间")
    created_time = models.DateTimeField(auto_now_add=True)
    supply_code = models.CharField(max_length=5, default="", verbose_name=u"补件编码,待补件状态才有补件编码")
