# coding=utf-8
from django.conf import settings

from .es import get_es, get_highlight, es_index_adapt


def search_tag(query='', offset=0, size=5):
    """
    @param query: 搜索词
    @param offset: 偏移量
    @param size: 返回个数

    tag(圈子)搜索
    搜索域:[
        1.tag名
        ]
    默认排序:[
        匹配度
        热度(标记了tag的所有帖子的点赞数+回复数)
        最后更新时间(标记了tag的所有帖子的编辑/回复时间)
        ]
    """
    # 参数验证
    size = min(size, settings.COUNT_LIMIT)

    # 搜索关键字部分
    # 搜索域
    multi_fields = {
        'name': 2,
        'description': 1, # 5.3版 去掉搜索域和高亮: 小组描述
                          # 5.9.2版 重新加上
    }
    fields = ['^'.join((k,str(v))) for (k,v) in multi_fields.iteritems()]

    multi_match = {
        'query': query,
        'type': 'cross_fields',
        'operator':'and',
        'fields': fields,
    }
    q = {
        'multi_match':multi_match,
    }

    # 过滤器部分
    f = [{'term':{'is_online':True}}] # 只返回上线的tag
    q = {
        'query':{'filtered':{
            'query':q,
            'filter':{'bool':{'must':f}}
        }}
    }

    # 排序规则部分
    q['sort'] = [
        '_score',
    ]

    # 高亮部分
    q['highlight'] = get_highlight(multi_fields.keys())

    es = get_es()
    index = es_index_adapt(
        index_prefix=settings.ES_INDEX_PREFIX,
        doc_type='doctor',
        rw='read'
    )
    res = es.search(
        index = index,
        doc_type = 'tag',
        timeout = settings.ES_SEARCH_TIMEOUT,
        body = q,
        from_ = offset,
        size = size)

    '''
    {
      "hits":{
        "hits":[
          {
            "sort":[...], //排序用数值
            "_source":{ //tag文档，参见mapping/tag.json
              "last_update_time":"2015-05-23T02:53:33+08:00",
              ...
            },
            "highlight":{ //高亮字段
              "name":[
                "<em>\u53cc\u773c\u76ae</em>"
              ]
            },
            "_id":"15" //tag id(请自行转数字)
          }
        ],
        "total":9, //结果总数
      },
    }
    '''
    return res
