# coding: utf-8
import time
from django.db import models
from gm_types.gaia import GROUPBUY_TYPE, GROUPBUY_STATUS

from utils.time_tools import datetime2timestamp


class GroupBuyTeam(models.Model):
    class Meta:
        verbose_name = u"拼团小组"
        app_label = 'api'
        db_table = 'api_groupbuy_team'
        index_together = [
            ['hash_id'],
            ["service_item_id", "status"],
            ["end_time"],
        ]

    hash_id = models.CharField(max_length=16, verbose_name=u'外部显示拼团小组ID', unique=True)
    min_user_number = models.IntegerField(verbose_name=u"拼团人数最小限制")
    left_user_number = models.IntegerField(verbose_name=u"拼团剩余坑位")
    current_user_number = models.IntegerField(verbose_name=u"当前参团用户数量(包括未付款)")
    gb_type = models.IntegerField(verbose_name=u"拼团类型", choices=GROUPBUY_TYPE)
    status = models.IntegerField(
        verbose_name=u"拼团状态", choices=GROUPBUY_STATUS, default=GROUPBUY_STATUS.GROUPBUY_STARTED
    )
    service_item_id = models.IntegerField(verbose_name=u"拼团的美购项目")
    price_id = models.IntegerField(verbose_name=u"价格信息ID")
    creator_user_id = models.IntegerField(verbose_name=u"创建者")
    start_time = models.DateTimeField(verbose_name=u"开始时间")
    end_time = models.DateTimeField(verbose_name=u"结束时间")
    form_id = models.CharField(verbose_name=u'推送消息form_id, 有效期7天', max_length=256,default="")
    hour4_notify=models.BooleanField(verbose_name=u"4小时未完成消息通知已标记",default=False)
    hour24_notify=models.BooleanField(verbose_name=u"24小时未完成消息通知标记",default=False)
    success_notify=models.BooleanField(verbose_name=u'拼团完成消息通知标记',default=False)
    success_time = models.DateTimeField(verbose_name=u"拼团成功时间", null=True)
    update_time = models.DateTimeField(verbose_name=u"最近修改时间", auto_now=True)
    create_time = models.DateTimeField(verbose_name=u"创建时间", auto_now_add=True)

    @property
    def countdown(self):
        if self.status == GROUPBUY_STATUS.GROUPBUY_STARTED:
            return max(datetime2timestamp(self.end_time) - int(time.time()), 0)
        return 0

    def to_dict(self):
        return {
            'id': self.id,
            'groupbuy_code': self.hash_id,
            'price_id': self.price_id,
            'min_user_number': self.min_user_number,
            'left_user_number': self.left_user_number,
            'gb_type': self.gb_type,
            'status': self.status,
            'service_item_id': self.service_item_id,
            'creator_user_id': self.creator_user_id,
            'countdown': self.countdown,
        }

    def to_display_dict(self):
        return {
            'id':self.id,
            'groupbuy_code': self.hash_id,
            'price_id': self.price_id,
            'min_user_number': self.min_user_number,
            'left_user_number': self.left_user_number,
            'creator_user_id':self.creator_user_id,
            'status': self.status,
            'service_item_id': self.service_item_id,
            'countdown': self.countdown,
            'members': [],
        }


class GroupBuyTeamOrder(models.Model):
    class Meta:
        verbose_name = u"拼团小组和订单的关联表"
        app_label = 'api'
        db_table = 'api_groupbuyteam_order'

    user_id = models.IntegerField(verbose_name=u"参与拼团用户ID")
    groupbuy_team = models.ForeignKey(GroupBuyTeam)
    order_id = models.CharField(verbose_name=u"订单ID", max_length=12, db_index=True)
    paid = models.BooleanField(verbose_name=u"是否支付", default=False)
