# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import time
import datetime

from celery import shared_task
from django.conf import settings

from api.models import Reservation, RESERVATION_TYPE
from sms.utils.smsfactory import send_sms
from api.tool.notification_tool import send_notification

from rpc.tool.protocol import PushUrlProtocol
from services.doctor_notify_service import DoctorNotifyService


@shared_task
def doctor_confirm_reservation(reservation):
    if not isinstance(reservation, Reservation):
        reservation = Reservation.objects.get(id=reservation)

    d = reservation.date
    datetime_str = u'{}月{}日 {:02}:{:02}'.format(d.month, d.day, d.hour, d.minute)
    content = settings.RESERVATION_CONFIRM_CONTENT.format(
        datetime_str,
        reservation.schedule.doctor.name,
    )
    send_notification(
        reservation.user.id,
        settings.RESERVATION_CONFIRM_TITLE,
        content,
        settings.PROTOCOL_MY_RESERVATION,
    )
    send_sms(reservation.order.phone, 14, [{'doctor_name': reservation.schedule.doctor.name}, {'time': datetime_str}])


@shared_task
def doctor_cancel_reservation(reservation):
    if not isinstance(reservation, Reservation):
        reservation = Reservation.objects.get(id=reservation)

    d = reservation.date
    datetime_str = u'{}月{}日 {:02}:{:02}'.format(d.month, d.day, d.hour, d.minute)
    content = settings.RESERVATION_CANCEL_CONTENT.format(
        datetime_str,
        reservation.schedule.doctor.name,
    )
    send_notification(
        reservation.user.id,
        settings.RESERVATION_CANCEL_TITLE,
        content,
        settings.PROTOCOL_MY_RESERVATION,
    )
    cancel_reason = reservation.cancel_reason
    if cancel_reason:
        send_sms(reservation.order.phone, 11, [
            {'doctor_name': reservation.schedule.doctor.name},
            {'time': datetime_str},
            {'reason': cancel_reason}
        ])
    else:
        send_sms(reservation.order.phone, 41,
                 [{'doctor_name': reservation.schedule.doctor.name}, {'time': datetime_str}])


@shared_task
def user_cancel_reservation(reservation):
    if not isinstance(reservation, Reservation):
        reservation = Reservation.objects.get(id=reservation)

    d = reservation.date
    datetime_str = u'{}月{}日 {:02}:{:02}'.format(d.month, d.day, d.hour, d.minute)
    reservation_type = RESERVATION_TYPE.getDesc(reservation.reservation_type)

    content = settings.RESERVATION_USER_CANCEL_CONTENT.format(
        reservation.user.last_name,
        datetime_str,
        reservation_type,
    )
    url = PushUrlProtocol.DOCTOR_RESERVATION_LIST.format(
        reservation.reservation_type,
        reservation.status,
    )

    send_notification(
        reservation.schedule.doctor.user_id,
        settings.RESERVATION_USER_CANCEL_TITLE,
        content,
        url,
    )

    ss = DoctorNotifyService(reservation.schedule.doctor_id)
    ss.notify_cancel_reservation(reservation)


@shared_task
def user_create_reservation(reservation):
    """ 用户创建预约成功后向医生发送通知
    """
    if not isinstance(reservation, Reservation):
        reservation = Reservation.objects.get(id=reservation)

    start_time = reservation.date
    time_delta = (datetime.timedelta(hours=1)
                  if reservation.reservation_type == RESERVATION_TYPE.SURGERY
                  else datetime.timedelta(minutes=30))
    end_time = reservation.date + time_delta

    datetime_str = u'{}月{}日，{:02}:{:02}到{:02}:{:02}'.format(
        start_time.month, start_time.day, start_time.hour,
        start_time.minute, end_time.hour, end_time.minute)
    content = settings.RESERVATION_USER_CREATE.format(
        reservation.user.last_name, datetime_str
    )
    url = PushUrlProtocol.DOCTOR_RESERVATION_LIST.format(
        reservation.reservation_type,
        reservation.status,
    )

    send_notification(
        reservation.schedule.doctor.user_id,
        settings.RESERVATION_USER_CREATE_TITLE,
        content,
        url,
    )

    # added at 593
    ss = DoctorNotifyService(reservation.schedule.doctor_id)
    ss.notify_new_reservation(reservation)
