#! /usr/bin/env python
# -*- coding: utf-8 -*-
import json
from django.db.models import Q

from gm_types.gaia import USER_GENDER

from rpc.decorators import bind, bind_context
from api.models.face import SkinResult
from api.tool.user_tool import get_user_from_context
from api.services.skin_service import (
    AcneService,
    HeiTouService,
    HeiYanQuanService,
    SkinAgeService,
    SkinOilService,
    SkinColorService,
    SmoothLevelService,
    ActWrinklesService,
    EyeBagService,
)


@bind_context('api/face_app/save_skin_result', with_context=True)
def create_skin_result(ctx, data, device_id, image_url):
    """
    保存用户测肤的结果
    :param ctx:
    :param data: 测肤结果
    :return:
    """

    user = get_user_from_context(ctx)
    if not data:
        return
    obj = SkinResult()
    obj.user_id = user.id if user else 0
    obj.device_id = device_id
    obj.image_url = image_url

    obj.gender = USER_GENDER.FEMALE if data.get('gender') == 'female' else USER_GENDER.MALE
    obj.total_score = str(data.get('total_score'))
    obj.skin_age = str(data.get('skin_age'))
    obj.skin_color_desc = str(data.get('skin_color_desc'))
    obj.skin_color_level = str(data.get('skin_color_level'))
    obj.skin_color_temperature = str(data.get('skin_color_temperature'))

    obj.equilibrium = str(data.get('skin_radar').get('equilibrium'))
    obj.health = str(data.get('skin_radar').get('health'))
    obj.smooth = str(data.get('skin_radar').get('smooth'))
    obj.tolerance = str(data.get('skin_radar').get('tolerance'))
    obj.uniform = str(data.get('skin_radar').get('uniform'))
    obj.young = str(data.get('skin_radar').get('young'))

    obj.oil_level = str(data.get('oil_level'))
    obj.blackhead_number = str(data.get('blackhead_number'))
    obj.blackhead_severity = str(data.get('blackhead_severity'))
    obj.tzone_oil_shine = str(data.get('tzone_oil_shine'))
    obj.cheek_oil_shine = str(data.get('cheek_oil_shine'))
    obj.chin_oil_shine = str(data.get('chin_oil_shine'))
    obj.pore_severity = str(data.get('pore_severity'))
    obj.coarseness = str(data.get('coarseness'))

    obj.pigmental = str(data.get('dark_circle').get('pigmental'))
    obj.vascular = str(data.get('dark_circle').get('vascular'))
    obj.structural = str(data.get('dark_circle').get('structural'))
    obj.severity = str(data.get('dark_circle').get('severity'))

    obj.acne_json = json.dumps(data.get('acne')) if data.get('acne') else '[]'
    obj.acne_severity = str(data.get('acne_severity'))
    obj.acne_desc = str(data.get('acne_desc'))

    obj.short_oil_level_advice = str(data.get('short_oil_level_advice'))
    obj.short_coarseness_advice = str(data.get('short_coarseness_advice'))
    obj.short_blackhead_advice = str(data.get('short_blackhead_advice'))
    obj.short_acne_advice = str(data.get('short_acne_advice'))
    obj.short_dark_circle_advice = str(data.get('short_dark_circle_advice'))

    obj.save()
    return {'id': obj.id}


@bind_context('api/face_app/get_skin_result', with_context=True)
def get_skin_result(ctx, device_id):
    user = get_user_from_context(ctx)

    user_id = user.id if user else 0
    query = Q(device_id=device_id)
    if user_id:
        query = Q(user_id=user_id)
    obj = SkinResult.objects.filter(query).order_by('-id').first()
    if not obj:
        return {}
    return {
        'id': obj.id,
        'total_score': obj.total_score,

        'skin_radar': obj.skin_radar,
        'dark_circle': obj.dark_circle,

        'gender': 'female' if obj.gender == USER_GENDER.FEMALE else 'male',
        'skin_age': obj.skin_age,
        'skin_color_level': obj.skin_color_level,
        'skin_color_desc': obj.skin_color_desc,
        'skin_color_temperature': obj.skin_color_temperature,
        'oil_level': obj.oil_level,
        'blackhead_number': obj.blackhead_number,
        'blackhead_severity': obj.blackhead_severity,
        'tzone_oil_shine': obj.tzone_oil_shine,
        'cheek_oil_shine': obj.cheek_oil_shine,
        'chin_oil_shine': obj.chin_oil_shine,
        'pore_severity': obj.pore_severity,
        'coarseness': obj.coarseness,

        'acne': obj.acne,
        'acne_severity': obj.acne_severity,
        'acne_desc': obj.acne_desc,

        'short_oil_level_advice': obj.short_oil_level_advice,
        'short_coarseness_advice': obj.short_coarseness_advice,
        'short_blackhead_advice': obj.short_blackhead_advice,
        'short_acne_advice': obj.short_acne_advice,
        'short_dark_circle_advice': obj.short_dark_circle_advice,

        'image_url': obj.image_url,
        'device_id': obj.device_id,
        'user_id': obj.user_id,
        'created_time': obj.created_time.strftime("%Y-%m-%d %H:%M:%S"),
    }


@bind("api/skin_check/conf")
def get_skin_conf_info(filter_data):
    """
    获取测肤后台配置信息
    :param filter_data:  不同类型筛选字段集合 {"skin_age": {}, "skin_oil": {}} ……
    :return:
    """
    result = {
        "skin_age": SkinAgeService.get_skin_age_info(filter_data.get("skin_age", {})),  # 肤龄
        "skin_oil": SkinOilService.get_skin_oil_info(filter_data.get("skin_oil", {})),  # 油脂
        "black_head": HeiTouService.get_heitou_info(filter_data.get("black_head", {})),  # 黑头
        "acne": AcneService.get_acne_info(filter_data.get("acne", {})),  # 痘痘
        "smooth": SmoothLevelService.get_smooth_level_info(filter_data.get("smooth", {})),  # 光滑度
        "dark_circle": HeiYanQuanService.get_dark_circle_info(filter_data.get("dark_circle", {})),  # 黑眼圈
        "skin_color": SkinColorService.get_skin_color_info(filter_data.get("skin_color", {})),  # 肤色
        "wrinkles": ActWrinklesService.get_wrinkles_info(filter_data.get("wrinkles", {})),  # 法令纹,
        "eye_bags": EyeBagService.get_eye_bag_info(filter_data.get("eye_bags", {})),  # 眼袋,
    }

    return result


@bind("api/skin_check/get_all_info")
def get_skin_check_all_info():
    result = {
        "smooth_info": SmoothLevelService.get_smooth_all_images_info(),
    }

    return result


@bind("api/skin_check/has_lastest_record", with_context=True)
def skin_check_has_lastest_record(ctx, device_id):
    """
    是否有测肤记录
    :param ctx:
    :param device_id:
    :return:
    """
    user = get_user_from_context(ctx)

    user_id = user.id if user else 0
    query = Q(device_id=device_id)

    if user_id:
        query = Q(user_id=user_id)

    status = SkinResult.objects.filter(query).exists()

    return {
        "status": True if status else False,
    }
