# coding=utf-8
from __future__ import unicode_literals, absolute_import, print_function

from django.db.models import Q
from gm_types.gaia import TAG_TYPE, DOCTOR_TYPE, ACTIVITY_TYPE_ENUM, SERVICE_SELL_TYPE

from api.models import (
    Service, ServiceVideo, CouponInfo,
    Special, ServiceItem,
    ImageRelatedService, Doctor, OrganizationImage)
from api.tool.image_utils import get_full_path
from api.tool.log_tool import logging_exception,info_logger
from api.tool.service_tool import sell_count_display
from api.tool.user_tool import get_user_from_context
from rpc.cache import service_score_tag_avg
from rpc.decorators import bind, cache_page, bind_context
from rpc.tool.error_code import gen, CODES

from api.manager import service_info_manager
from api.manager import groupbuy_manager
from .goods_service.service_utils import get_show_special_info, get_service_header_pic_info, calcate_sku_discount_price


@bind_context('api/service/base_service_info')
def get_base_service_info(ctx, service_id, coupon_info_id=None, support_groupbuy=False):
    # 这里添加support_groupbuy参数, 默认为False, 对ship端接口调用表现一致,backend端设置参数为True, 拼团部分不做灰度控制
    try:
        service = Service.objects.get(id=service_id)
    except:
        return gen(CODES.SERVICE_NOT_EXSIT)

    user = get_user_from_context(ctx)
    data = service.get_base_service_detail(user, support_groupbuy)

    coupon_info = None

    if coupon_info_id:
        ci = CouponInfo.objects.filter(id=coupon_info_id).first()
        if ci and ci.user == user:
            coupon_info = ci

    multiattributes = service.get_multiattributes(coupon_info)

    data['multiattributes'] = multiattributes
    # 拼团部分不做灰度控制
    if support_groupbuy:
        sku_info = service.get_sku_detail_info(coupon_info)
        data['sku_info'] = sku_info

    special_list, seckill_list = [], []
    special_info_dict = service.get_can_sell_item_ids_and_current_special_info()
    for siid, sp_info in special_info_dict.items():
        if not sp_info.get('id'):
            continue

        if sp_info and sp_info.get('activity_type') == ACTIVITY_TYPE_ENUM.SECKILL:
            seckill_list.append(sp_info['id'])
        else:
            # special_list.append(sp_info['id'])  # 暂时先隐藏
            pass

    if len(seckill_list) > 0:
        data['special_id'] = max(seckill_list)
        s = Special.objects.get(id=data['special_id'])
        data['special_info'] = s.title
        data['is_seckill'] = s.is_seckill
        data['is_new_special'] = False
        data['special_seckill_type'] = s.seckill_type
    elif len(special_list) > 0:
        data['special_id'] = max(special_list)
        s = Special.objects.get(id=data['special_id'])
        data['special_info'] = s.title
        data['is_seckill'] = s.is_seckill
        data['is_new_special'] = s.is_new_special
        data['special_seckill_type'] = ''
    else:
        data['special_id'] = ''
        data['special_info'] = ''
        data['is_new_special'] = False
        data['is_seckill'] = False
        data['special_seckill_type'] = ''
    data['header_pic'] = []
    data_video = ServiceVideo.objects.filter(service=service_id)
    if data_video:
        video_info = data_video[0].get_video_info()
        data['header_pic'].append({'video_url':video_info.get('video_url'),'pic':video_info.get('video_pic')})
    data['header_pic'].append({'video_url':'','pic':data['pic']})
    return data

@bind_context('api/service/base_service_info_v2')
def get_base_service_info_v2(ctx, service_id, coupon_info_id=None):
    '''
    since: 7710
    获取基础美购信息---
    '''
    user = get_user_from_context(ctx)
    try:
        service = Service.objects.get(id=service_id)
    except:
        return gen(CODES.SERVICE_NOT_EXSIT)

    support_groupbuy = service.service_type != SERVICE_SELL_TYPE.FLAGSHIP_STORE_SERVICE
    data = service.get_base_service_detail(user, support_groupbuy=support_groupbuy)

    special_info = get_show_special_info(service)
    data.update(special_info)
    data['header_pic'] = get_service_header_pic_info(service)[:6]
    return data

@bind('api/service/get_service_feature')
def get_service_features(service_id):
    service = Service.objects.get(id=service_id)
    return service.get_service_features()


@bind('api/service/get_service_base_comment')
def get_service_base_comment(service_id):
    return {
        'count': '0',
        'tags': [],
        'comment': []
    }

@bind('api/service/get_group_buy_rolling_info')
def get_group_buy_rolling_info(service_id):
    '''
    获取美购详情页中 基于sku的跑马灯信息 及 基于全场的拼团成功提示
    '''
    unfinished_groupbuy_team_info = groupbuy_manager.get_groupbuy_unfinished(None, service_id=service_id)
    un_format_list = []
    for k, v in unfinished_groupbuy_team_info.items():
        un_format_list.append({
            'service_item_id': k,
            'groupbuy_teams': v
        })
    suc_top_show_info = groupbuy_manager.get_groupbuy_success_show_info()
    result = {
        'groupbuy_info_rolling': un_format_list,
        'groupbuy_suc_top': suc_top_show_info
    }
    return result

@bind('api/service/get_doctor_info')
@cache_page(60 * 5)
def get_service_doctor_info(service_id):
    service = Service.objects.select_related('doctor__hospital').get(id=service_id)
    return service.get_doctor_info()

@bind('api/service/get_hospital_doctor_info')
@cache_page(60 * 5)
def get_service_hospital_and_doctor_info(service_id):
    try:
        service = Service.objects.select_related('doctor__hospital').get(id=service_id)
        return service.get_hospital_and_doctor_info()
    except:
        return gen(CODES.SERVICE_NOT_EXSIT)


@bind('api/service/get_related_service')
def get_related_service(service_id):
    data = []
    service = Service.objects.get(id=service_id)
    recommand_service = service.get_recommend_services_object().exclude(recommend_service_id=service_id)
    exclude_ids = []
    service_id_list = service_info_manager.add_double_eleven_image()

    for recommand in recommand_service:
        service = recommand.recommend_service
        double_eleven_image = ''
        if service.id in service_id_list:
            double_eleven_image = 'https://heras.igengmei.com/serviceactivity/2019/10/21/d9c715646c'
        d = {
            'service_id': service.id,
            'service_name': service.project_type.name if service.project_type else service.name,
            'prices': service.gengmei_price_display,
            'is_multiattribute': service.is_multiattribute,
            'pic': get_full_path(service.image_header, '-half'),
            'is_recommend': True,
            'double_eleven_image': double_eleven_image,
        }
        exclude_ids.append(service.id)
        if service.is_multiattribute:
            d['prices'] = d['prices'].split('-')[0]
        data.append(d)

    services = Service.objects.filter(doctor=service.doctor,
                                      is_online=True).exclude(id__in=exclude_ids).exclude(id=service_id)
    services = services[0:9 - len(data)]
    for service in services:
        double_eleven_image = ''
        if service.id in service_id_list:
            double_eleven_image = 'https://heras.igengmei.com/serviceactivity/2019/10/21/d9c715646c'
        d = {
            'service_id': service.id,
            'service_name': service.project_type.name if service.project_type else service.name,
            'prices': service.gengmei_price_display,
            'is_multiattribute': service.is_multiattribute,
            'pic': get_full_path(service.image_header, '-half'),
            'is_recommend': False,
            'double_eleven_image': double_eleven_image,
        }
        if service.is_multiattribute:
            d['prices'] = d['prices'].split('-')[0]
        data.append(d)
    return data


@bind('api/service/get_related_service_hybrid')
def get_related_service_hybrid(service_id, start_num, count):
    service = Service.objects.get(id=service_id)
    services = Service.objects.filter(doctor=service.doctor, is_online=True)
    services = services[start_num:start_num + count]
    data = []
    for service in services:
        data.append({
            'id': service.id,
            'image': get_full_path(service.image_header, '-half'),
            'desc': service.name,
            'price': min(service.gengmei_price_list),
            'is_price_range': service.is_multiattribute,
            'appoint_count': sell_count_display(service.sell_amount_display),
            'label': service.label,
            'is_soldout': service.is_can_be_sold()
        })
    return data


@bind('api/service/guess_you_like')
def get_guess_you_like(service_id, start_num, count, user_city_tag_id=None, user_city_id=None):
    service = Service.objects.select_related('doctor').get(id=service_id)

    hospital_id = service.doctor.hospital_id if service.doctor and service.doctor.hospital_id else None

    if hospital_id is None:
        return []

    tags = service.tags.all()
    tag_list = []
    data = []
    for tag in tags:
        if tag.tag_type == TAG_TYPE.BODY_PART_SUB_ITEM:
            tag_list.append(tag.id)
        if tag.tag_type == TAG_TYPE.ITEM_WIKI:
            p_tags = tag.parent_tags()
            for p_tag in p_tags:
                if p_tag.tag_type == TAG_TYPE.BODY_PART_SUB_ITEM:
                    tag_list.append(p_tag.id)

    try:
        from rpc.context import get_rpc_remote_invoker

        # max_size = 20
        query_size = count + 5  # 多捞几个，避免几种不同方式里面有当前service_id导致捞不足够max_size

        tag_id = None
        if len(tag_list) > 0:
            tag_id = tag_list[0]

        user_city_info = {}
        if user_city_tag_id and user_city_id:
            user_city_info = {
                "user_city_tag_id": user_city_tag_id,
                "user_city_id": user_city_id,
            }

        result = get_rpc_remote_invoker()['doris/search/guess_you_like_service'](
            hospital_id=hospital_id,
            tag_id=tag_id, user_city_info=user_city_info,
            offset=start_num, size=query_size).unwrap()

        # 去掉潜在重复的id（当前美购id），然后把冗余的数据去掉，只要前20条
        result_ids = [sid for sid in result['service_ids'] if sid != service_id][:20]
        service_ids = result_ids[start_num: start_num + count]

        sid_to_service_info = service_info_manager.get_toc_spu_info_list_mapping_by_spu_ids(service_ids)

        for sid in service_ids:
            if str(sid) in sid_to_service_info:
                si = sid_to_service_info[str(sid)]
                try:
                    data.append({
                        'id': si["service_id"],
                        'image': get_full_path(si["image_header"], '-half'),
                        'desc': si["service_name"],
                        'price': si["seckill_price"] if si["is_seckill"] else si["gengmei_price"],
                        'is_price_range': si["is_price_range"],
                        'appoint_count': sell_count_display(si["sell_amount"]),
                        'label': si["label"],
                        'is_soldout': si["is_can_be_sold"],
                        'double_eleven_image': si.get("double_eleven_image") or '',
                    })
                except:
                    logging_exception()
        return data
    except:
        logging_exception()
        return []


@bind('api/service/get_album_pic')
def get_service_album(service_id, start_num, count=10):
    """
    返回美购对应的相册图片
    """
    result = []
    service = Service.objects.get(id=service_id)
    image_pic = ImageRelatedService.objects.filter(service_id=service_id)
    image_pic_num = image_pic.count()
    doctor = service.doctor
    if doctor.doctor_type == DOCTOR_TYPE.OFFICER:
        doctor_pics = Doctor.objects.filter(hospital_id=doctor.hospital_id,
                                            is_online=True).filter(~Q(portrait=''))
    else:
        doctor_pics = Doctor.objects.filter(id=doctor.id,
                                            is_online=True).filter(~Q(portrait=''))
    doctor_pic_count = doctor_pics.count()
    organization_pics = OrganizationImage.objects.filter(doctor=doctor)

    if start_num < image_pic_num:
        '''先取其他里面的图片'''
        delta_num = image_pic_num - start_num
        if delta_num >= count:
            pics = image_pic[start_num: start_num + count]
            for pic in pics:
                result.append({
                    'pic': get_full_path(pic.image_url, '-half'),
                    'pic_w': get_full_path(pic.image_url, '-w'),
                    'name': u"其他"
                })
            return result
        else:
            pics = image_pic[start_num: start_num + count]
            for pic in pics:
                result.append({
                    'pic': get_full_path(pic.image_url, '-half'),
                    'pic_w': get_full_path(pic.image_url, '-w'),
                    'name': u"其他"
                })
            count = count - delta_num

    if start_num < doctor_pic_count and count > 0:
        '''首先获取医生图片'''
        pics = doctor_pics[start_num:start_num + count]
        for pic in pics:
            result.append({
                'pic': get_full_path(pic.portrait, '-half'),
                'pic_w': get_full_path(pic.portrait, '-w'),
                'name': u"医生 " + pic.name
            })

        '''医生数量不够 用机构图片来弥补'''
        if len(result) < count:
            num = count - len(result)
            pics = organization_pics[0:num]
            for pic in pics:
                result.append({
                    'pic': get_full_path(pic.image_url, '-half'),
                    'pic_w': get_full_path(pic.image_url, '-w'),
                    'name': u"环境"
                })
    else:
        start_num = start_num - doctor_pic_count
        pics = organization_pics[start_num:start_num + count]
        for pic in pics:
            result.append({
                'pic': get_full_path(pic.image_url, '-half'),
                'pic_w': get_full_path(pic.image_url, '-w'),
                'name': u"环境"
            })

    return result


@bind('api/service/get_buy_notice')
def get_buy_notice():
    from api.tool.service_tool import ServiceConfig
    return {"buy_notice": ServiceConfig.get()['config_content']}


@bind('api/service/get_service_score_info')
def get_service_score_info(service_id):
    s = Service.objects.get(id=service_id)
    tag = s.project_type
    avg_score = service_score_tag_avg.get(str(tag.id))
    if avg_score is None:
        avg_score = 0
    else:
        avg_score = float(avg_score)
    if s.rating > avg_score:
        desc = '非常好'
        type = 0
    elif s.rating == avg_score:
        desc = '不错'
        type = 0
    else:
        desc = '一般'
        type = 1

    if s.rating:
        return {
            'total_score': round(s.rating, 1),
            'total_desc': desc,
            'type': type,
            'score_detail': [{
                'score': s.operation_effect_rating,
                'desc': '术后效果',
            }, {
                'score': s.hospital_env_rating,
                'desc': '就医环境'
            }, {
                'score': s.doctor_attitude_rating,
                'desc': '医生态度',
            }]
        }
    else:
        return {}
