# -*- coding: UTF-8 -*-

from django.db import models
from gm_types.gaia import DOCTOR_BULLETIN_SHOW_AREA

from api.tool.datetime_tool import get_timestamp
from gm_upload import IMG_TYPE, ImgUrlField


class Content(models.Model):
    """医生运营内容"""
    class Meta:
        app_label = 'doctor'

    title = models.CharField(u'列表页标题', max_length=50)
    content = models.TextField(u'内容')
    created_time = models.DateTimeField(u'创建时间', auto_now=True)
    updated_time = models.DateTimeField(u'最后更新时间', auto_now_add=True)


class HomePopup(models.Model):
    """首页弹窗"""
    class Meta:
        app_label = 'doctor'

    title = models.CharField(u'标题', max_length=50)
    image = ImgUrlField(u'图片', img_type=IMG_TYPE.HOMEPOPUP, max_length=200, default='')
    content = models.ForeignKey('Content', verbose_name=u'医生运营内容')
    is_online = models.BooleanField(u'是否上线', default=False)
    begin_time = models.DateTimeField(u'开始时间')
    end_time = models.DateTimeField(u'结束时间')
    created_time = models.DateTimeField(u'创建时间', auto_now_add=True)
    updated_time = models.DateTimeField(u'最后更新时间', auto_now=True)

    def to_data(self):
        return {
            'image': self.image,
            'content_id': self.content and self.content.id,
        }


class HomeSlide(models.Model):
    """首页轮播图"""
    class Meta:
        app_label = 'doctor'

    title = models.CharField(u'标题', max_length=50)
    image = ImgUrlField(u'图片', img_type=IMG_TYPE.HOMESLIDE, max_length=200, default='')
    ordering = models.IntegerField(u'排序', default=0)
    content = models.ForeignKey('Content', verbose_name=u'医生运营内容')
    is_online = models.BooleanField(u'是否上线', default=False)
    begin_time = models.DateTimeField(u'开始时间')
    end_time = models.DateTimeField(u'结束时间')
    created_time = models.DateTimeField(u'创建时间', auto_now_add=True)
    updated_time = models.DateTimeField(u'最后更新时间', auto_now=True)

    def to_data(self):
        return {
            'image': self.image,
            'content_id': self.content and self.content.id,
            'title': self.title if self.title else ''
        }


class Bulletin(models.Model):
    """公告"""
    class Meta:
        app_label = 'doctor'

    title = models.CharField(u'标题', max_length=50)
    image = ImgUrlField(u'图片', img_type=IMG_TYPE.BULLETIN, max_length=200, default='')
    content = models.ForeignKey('Content', verbose_name=u'医生运营内容')
    is_online = models.BooleanField(u'是否上线', default=False)
    online_time = models.DateTimeField(u'更新上线时间', null=True)
    created_time = models.DateTimeField(u'创建时间', auto_now_add=True)
    updated_time = models.DateTimeField(u'最后更新时间', auto_now=True)

    def to_data(self):
        return {
            'id': self.id,
            'title': self.title,
            'date': get_timestamp(self.online_time),
            'image': self.image,
            'content_id': self.content and self.content.id,
            'content': self.content.content,
        }


class BulletinExtra(models.Model):
    """
    公告展示范围
    """
    bulletin = models.ForeignKey(Bulletin, verbose_name='公告')
    city_id = models.CharField(max_length=40, verbose_name='展示城市的id', blank=True, null=True)
    merchant_id = models.BigIntegerField(verbose_name=u'商户id', blank=True, null=True)
    area = models.CharField(max_length=16, choices=DOCTOR_BULLETIN_SHOW_AREA, default=DOCTOR_BULLETIN_SHOW_AREA.COUNTRY)
    created_time = models.DateTimeField(u'创建时间', auto_now_add=True)


class BulletinRead(models.Model):
    """
        医生已读公告记录
    """
    class Meta:
        app_label = 'doctor'
        unique_together = ('doctor_id', 'bulletin')

    doctor_id = models.CharField(verbose_name=u'医生id', max_length=100)
    bulletin = models.ForeignKey(Bulletin, verbose_name=u'公告')
