# -*- coding: UTF-8 -*-

from django.db.models import Q
from gm_types.gaia import DOCTOR_TYPE, RESERVATION_STATUS
from gm_dataquery.db import DB

from gm_dataquery.dataquery import DataBuilder, DataSQLQuery
from api.tool.verification_code import VerifiedCode
from api.models import VERIFY_CODE_TYPE
from api.models import Doctor, Reservation, DoctorAccount


class DoctorDB(DataBuilder):
    def getval_person_phone(self, obj):
        try:
            phone = obj.user.person.phone
        except:
            phone = ''
        return phone

    def getval_validate_code(self, obj):
        try:
            phone = obj.user.person.phone
        except:
            phone = ''

        return VerifiedCode.generate(phone, VERIFY_CODE_TYPE.LOGIN) if phone else ''

    def getval_hospital__name(self, obj):
        return obj.hospital.name if getattr(obj, 'hospital', None) else ''

    def getval_doctor_conversation(self, obj):
        return obj.id if obj.user else ''  # 未绑定用户

    def getval_reservation(self, obj):
        return Reservation.objects.filter(schedule__doctor=obj, status=RESERVATION_STATUS.RESERVING).count()

    def getval_reservation_expired(self, obj):
        return Reservation.objects.filter(schedule__doctor=obj, status=RESERVATION_STATUS.EXPIRED).count()

    def getval_doctor_type(self, obj):
        if obj.doctor_type:
            return DOCTOR_TYPE.getDesc(obj.doctor_type)

    def getval_accountstatus(self, obj):
        return obj.account.status if obj.account else 0

    def getval_machine_code(self, obj):
        if obj.pos.exists():
            return obj.pos.first().code

    def getval_machine_id(self, obj):
        if obj.pos.exists():
            return obj.pos.first().id


@DB
class DoctorDQ(DataSQLQuery):
    model = Doctor
    data_model = DoctorDB

    def filter_online_time(self, srch_key, srch_val, regex=False):
        return self._qry_time_range(srch_key, srch_val, regex)

    def filter_created_at(self, srch_key, srch_val, regex=False):
        return self._qry_time_range(srch_key, srch_val, regex)

    def filter_share_topic_num(self, srch_key, srch_val, regex=False):
        if srch_val == '0':
            return Q(share_topic_num=srch_val)
        else:
            return Q(share_topic_num__gte=srch_val)

    def filter_has_online_services(self, srch_key, srch_val, regex=False):
        if srch_val == '':
            return None
        self.distinct = True
        q = Q(services__isnull=False)
        if srch_val == '1':
            return q
        else:
            return ~q

    def filter_services__is_online(self, srch_key, srch_val, regex=False):
        if srch_val == '':
            return None
        self.distinct = True
        q = Q(services__is_online=True)
        if srch_val == '1':
            return q
        else:
            return ~q

    def filter_is_maidan(self, srch_key, srch_val, regex=False):
        if srch_val == '':
            return None
        q = Q(maidansetting__is_open=True)
        if srch_val == '1':
            return q
        else:
            return ~q

    def filter_last_edit_time(self, srch_key, srch_val, regex=False):
        return self._qry_time_range(srch_key, srch_val, regex)

    def filter_accountstatus(self, srch_key, srch_val, regex=False):
        return Q(d_merchant__account__status=srch_val)

    def filter_doctor_status(self, srch_key, srch_val, regex=False):
        if srch_val == '7':
            return Q(status=7)
        else:
            return ~Q(status=7)

    def filter_with_punishment(self, key, val, regex=False):
        return Q(punishment__isnull=not bool(int(val)))
