# -*- coding: UTF-8 -*-
from __future__ import absolute_import

from django.db.models import Q
from gm_dataquery.db import DB
from gm_dataquery.dataquery import DataBuilder, DataSQLQuery
from gm_types.gaia import (
    MULTITOPIC_TYPE, MULTITOPIC_SERVICE_TYPE, MULTITOPIC_REVIEW_TYPE,
    MULTITOPIC_REVIEW_STATUS,
)

from api.models import MultiTopic, ZoneWhiteList, MultiTopicReview
from answer.models import Question, QuestionImage
from talos.models.topic import Problem, TopicImage


class MultitopicDB(DataBuilder):
    def getval_multitopic_type(self, obj):
        return MULTITOPIC_TYPE.getDesc(obj.multitopic_type)

    def getval_service_type(self, obj):
        return MULTITOPIC_SERVICE_TYPE.getDesc(obj.service_type)


@DB
class MultitopicDQ(DataSQLQuery):
        model = MultiTopic
        data_model = MultitopicDB

        def filter_effect_time(self, srch_key, srch_val, regex=False):
            return self._qry_time_range(srch_key, srch_val, regex)

        def filter_end_time(self, srch_key, srch_val, regex=False):
            return self._qry_time_range(srch_key, srch_val, regex)


class ZonewhitelistDB(DataBuilder):
    def getval_zone_name(self, obj):
        return obj.zone.tag.name


@DB
class ZonewhitelistDQ(DataSQLQuery):
    model = ZoneWhiteList
    data_model = ZonewhitelistDB

    def filter_zone_name(self, srch_key, srch_val, regex=False):
        return Q(zone__tag__name__contains=srch_val)

    def filter_zone_id(self, srch_key, srch_val, regex=False):
        return Q(zone_id=srch_val)


class MultitopicreviewDB(DataBuilder):
    def getval_content(self, obj):
        if obj.content_type == MULTITOPIC_REVIEW_TYPE.PROBLEM:
            problem = Problem.objects.get(pk=obj.content_id)
            return problem.answer[:100] + '...' if len(problem.answer) > 100 else problem.answer
        else:
            question = Question.objects.get(pk=obj.content_id)
            return question.content[:100] + '...' if len(question.content) > 100 else question.content

    def getval_content_desc(self, obj):
        return MULTITOPIC_REVIEW_TYPE.getDesc(obj.content_type)

    def getval_created_time(self, obj):
        if obj.content_type == MULTITOPIC_REVIEW_TYPE.PROBLEM:
            problem = Problem.objects.get(pk=obj.content_id)
            return problem.created_time.strftime("%Y-%m-%d %H:%M:%S")
        else:
            question = Question.objects.get(pk=obj.content_id)
            return question.create_time.strftime("%Y-%m-%d %H:%M:%S")

    def getval_recommend(self, obj):
        if obj.status == MULTITOPIC_REVIEW_STATUS.PASS:
            return u'是'
        else:
            return u'否'

    def getval_is_online(self, obj):
        if obj.content_type == MULTITOPIC_REVIEW_TYPE.PROBLEM:
            problem = Problem.objects.get(pk=obj.content_id)
            return problem.is_online
        else:
            question = Question.objects.get(pk=obj.content_id)
            return question.is_online

    def getval_content_have_images(self, obj):
        if obj.content_type == MULTITOPIC_REVIEW_TYPE.PROBLEM:
            flag = TopicImage.objects.filter(topic_id=obj.content_id).count()
            return True if flag else False
        else:
            flag = QuestionImage.objects.filter(question_id=obj.content_id).count()
            return True if flag else False

    def getval_user_name(self, obj):
        if obj.content_type == MULTITOPIC_REVIEW_TYPE.PROBLEM:
            problem = Problem.objects.get(pk=obj.content_id)
            return problem.user.nickname
        else:
            question = Question.objects.get(pk=obj.content_id)
            return question.user.nickname


@DB
class MultitopicreviewDQ(DataSQLQuery):
    model = MultiTopicReview
    data_model = MultitopicreviewDB

    def filter_user_id(self, srck_key, srck_val, regex=False):
        problem_ids = Problem.objects.filter(user_id=srck_val).values_list('id', flat=True)
        question_ids = Question.objects.filter(user=srck_val).values_list('id', flat=True)
        content_ids = set(problem_ids) | set(question_ids)
        return Q(content_id__in=content_ids)
