# -*- coding: UTF-8 -*-

import json
from django.db.models import Q
from gm_dataquery.dataquery import DataBuilder, DataSQLQuery
from gm_dataquery.db import DB

from gm_types.gaia import (
    SLIDE_PAYMENT_TYPE,
    SLIDE_USER_TYPE,
    SERVICE_HOME_OPERATION,
    SERVICE_DISPLAY_PORT,
    FONT_COLOR,
    SERVICEHOME_OPERATION_POSITION
)


from api.models import (
    SpecialItem,
    ServiceFilter,
    CategoryGadget,
    NewCategory,
    ServiceHomeBackground,
    ServiceHomeOperation,
    ServiceRelatedOperation,
)
from rpc.tool.dict_mixin import to_dict



class RecommendDB(DataBuilder):
    def getval_service__id(self, obj):
        return obj.service.id

    def getval_service__name(self, obj):
        return obj.service.name

    def getval_service__is_online(self, obj):
        if obj.service.is_online:
            return '是'
        else:
            return '否'

    def getval_pre_payment_price(self, obj):
        service = obj.service
        pre_payment_price = service.pre_payment_price
        if service.is_multiattribute:
            pre_payment_price = service.items.aggregate(c=Min('pre_payment_price')).get('c')
        return pre_payment_price

    def getval_gengmei_price(self, obj):
        service = obj.service
        gengmei_price = service.gengmei_price
        if service.is_multiattribute:
            gengmei_price = service.items.aggregate(c=Min('gengmei_price')).get('c')
        return gengmei_price

    def getval_discount(self, obj):
        service = obj.service
        discount = service.discount
        if service.is_multiattribute:
            discount = service.items.aggregate(c=Min('discount')).get('c')
        return discount

    def getval_is_online(self, obj):
        return obj.service.is_online


class RecommendDQ(DataSQLQuery):
        model = SpecialItem
        data_model = RecommendDB


class ServiceFilterDB(DataBuilder):
    pass


@DB
class ServiceFilterDQ(DataSQLQuery):
        model = ServiceFilter
        data_model = ServiceFilterDB


class CategoryGadgetDB(DataBuilder):
    def getval_image_data(self, obj, need_escape=False):
        return json.loads(obj.image_data)


@DB
class CategoryGadgetDQ(DataSQLQuery):
    model = CategoryGadget
    data_model = CategoryGadgetDB


class NewCategoryDB(DataBuilder):
    def getval_cities_name(self, obj):
        return [x.name for x in obj.cities.all()]

    def getval_regions_name(self, obj):
        return [x.name for x in obj.regions.all()]


@DB
class NewCategoryDQ(DataSQLQuery):
    model = NewCategory
    data_model = NewCategoryDB

    def create(self, **kwargs):
        tags = kwargs.pop('tags')
        cities = kwargs.pop('cities')
        regions = kwargs.pop('regions')
        obj = self.model.objects.create(**kwargs)
        obj.tags = tags
        obj.cities = cities
        obj.regions = regions
        return to_dict(obj)


class ServiceHomeBackgroundDB(DataBuilder):
    pass


@DB
class ServiceHomeBackgroundDQ(DataSQLQuery):
    model = ServiceHomeBackground
    data_model = ServiceHomeBackgroundDB


class ServiceHomeOperationDB(DataBuilder):
    def getval_details(self, obj, need_escape=False):
        return json.loads(obj.details)

    def getval_cities_name(self, obj):
        return [x.name for x in obj.cities.all()]

    def getval_regions_name(self, obj):
        return [x.name for x in obj.regions.all()]

    def getval_operation_type_desc(self, obj):
        return SERVICE_HOME_OPERATION.getDesc(obj.operation_type)

    def getval_user_type_desc(self, obj):
        return SLIDE_USER_TYPE.getDesc(obj.user_type)

    def getval_payment_type_desc(self, obj):
        return SLIDE_PAYMENT_TYPE.getDesc(obj.payment_type)

    def getval_position(self, obj):
        return sorted(list(ServiceRelatedOperation.objects.filter(servicehome_id=obj.id).values_list("position", flat=True)))


@DB
class ServiceHomeOperationDQ(DataSQLQuery):
    model = ServiceHomeOperation
    data_model = ServiceHomeOperationDB

    def filter_position(self, srch_key, srch_val, regex=False):
        service_home_ids = ServiceRelatedOperation.objects.filter(position__in=srch_val).values_list('servicehome_id', flat=True)
        if service_home_ids:
            return Q(id__in = service_home_ids)
        return Q()

