# coding=utf8

from django.conf import settings

from api.models import City, TransactionNearbyCity, TransactionNearbyCityToES, Region
from api.models import Tag


def get_nearby_province_tag(city_id):
    """
    获取临近省份tag信息
    :param city_id:
    :return: [('tag_id': int, 'name': string)]
    """
    try:
        province_id = City.objects.get(id=city_id).province_id
    except City.DoesNotExist:
        return []

    nearby_province_ids = settings.NEARBY_REGION.get(province_id, ())
    nearby_province_ids = nearby_province_ids + (province_id,)
    result = City.objects.filter(
        province_id__in=nearby_province_ids
    ).exclude(id=city_id).values_list('tag_id', 'name')
    return [{'tag_id': x[0], 'name': x[1]} for x in result]


def get_nearby_city_tag(city_id):
    """
    美购医生医院关于邻近城市的获取逻辑
    1、从邻近城市表中取数据，然后从city表中捞对应city的数据
    2、邻近城市表中取不到数据，就把该城市所在的省份以及所在省份的邻近省份的所有城市当作该城市的邻近城市，从city表中捞数据
    三处地方使用
    diary_transfer sku_transfer service_transfer
    :param city_id:
    :return: [('tag_id': int, 'name': string)]
    """
    nearby_city_ids = TransactionNearbyCityToES.objects.filter(city_id=city_id).values_list('nearby_city', flat=True)
    result = City.objects.filter(id__in=nearby_city_ids).values_list('tag_id', 'name')
    return [{'tag_id': x[0], 'name': x[1]} for x in result]


def get_nearby_city_tag_ids(city_id):
    """
    :param city_id:
    :return tag_id list
    """
    nearby_city_ids = TransactionNearbyCityToES.objects.filter(city_id=city_id).values_list('nearby_city', flat=True)
    city_names = City.objects.filter(id__in=nearby_city_ids).values_list('name', flat=True)
    tag_list = Tag.objects.filter(name__in=city_names).values_list('id', flat=True)
    return list(tag_list)


class NearbyCityGraph(object):
    def __init__(self):
        self.cityMapping = {}

    def addEdge(self, v, w):
        if self.cityMapping.get(v) is None:
            self.cityMapping[v] = []
        self.cityMapping[v].append(w)

    def get_tags(self, city_id):
        city_ids = self.cityMapping.get(city_id, [])
        cities = City.objects.filter(id__in=city_ids).values_list('tag_id', 'name')
        return [{'tag_id': x[0], 'name': x[1]} for x in cities]

    def get_tag_ids(self, city_id):
        city_ids = self.cityMapping.get(city_id, [])
        city_names = City.objects.filter(id__in=city_ids).values_list('name', flat=True)
        tag_ids = Tag.objects.filter(name__in=city_names).values_list('id', flat=True)
        return list(tag_ids)

    @classmethod
    def build(cls):
        """
        创建有向图
        :return:
        """
        g = NearbyCityGraph()
        city_ids = City.objects.all().values_list('id', flat=True)
        related_cities = TransactionNearbyCity.objects.all().values_list('city_id', 'nearby_city_id')
        for related_city in related_cities:
            g.addEdge(related_city[1], related_city[0])

        for city_id in city_ids:
            province_id = City.objects.get(id=city_id).province_id
            nearby_province_ids = settings.NEARBY_REGION.get(province_id, ())
            nearby_province_ids = nearby_province_ids + (province_id,)
            province_nearby_city_ids = City.objects.filter(province_id__in=nearby_province_ids)\
                .exclude(id=city_id).values_list('id', flat=True)
            hit_city_ids = TransactionNearbyCity.objects.all().values_list('city_id', flat=True)
            miss_city_ids= list(set(province_nearby_city_ids) - set(hit_city_ids))
            for miss_city_id in miss_city_ids:
                g.addEdge(city_id, miss_city_id)
        return g


def get_region_by_city_id(city_id=None):
    """
    根据城市id获取所在大区
    :param city_id: 没有城市 返回华南大区 http://wiki.wanmeizhensuo.com/pages/viewpage.action?pageId=28916990
    :return:
    """
    region = Region.objects.get(english_name='huanan')
    if not city_id:
        return {'region_id': region.id}
    try:
        area = City.objects.select_related('province').get(id=city_id)
    except City.DoesNotExist:
        return {'region_id': region.id}

    return {'region_id': area.province.region_id}
