# coding=utf8

from __future__ import unicode_literals, absolute_import, print_function

from helios.rpc import RPCFaultException
from gm_types.gaia import USER_RIGHTS_LEVEL
from django.conf import settings

from talos.libs.utils import convert_to_type_or
from talos.libs.image_utils import get_thumb_path
from talos.services.models.base import ServiceModelBase
from talos.backbone.rpc import RPCMixin


class User(RPCMixin, ServiceModelBase):
    """user object.

    properties:
        @id int
        @person_id str
        @nickname str
        @portrait(with -thumb) str

        @topic_count: online topic count int
        @vote_count: votes count received int

        @membership_level str
        @city_id str
        @city_name str
        @city_tag_id int
        @province_tag_id int
        @country_tag_id int
    """

    __slots__ = [
        'id', 'person_id', 'nickname', 'portrait',
        'topic_count', 'vote_count', 'membership_level',
        'current_city_id', 'city_id', 'city_name', 'city_tag_id',
        'province_tag_id', 'country_tag_id', 'user_rights_level', 'constellation', 'membership_icon',
        'level_icon', 'constellation_icon',
    ]

    def __init__(self, id, person_id, nickname, portrait,
                 topic_count=0, vote_count=0, membership_level=None,
                 current_city_id=None, city_id=None, city_name=None, city_tag_id=None,
                 province_tag_id=None, country_tag_id=None, user_rights_level=USER_RIGHTS_LEVEL.V1,
                 constellation=None, membership_icon=None, level_icon=None, constellation_icon=None):
        self.id = convert_to_type_or(id, int, None)
        self.person_id = person_id
        self.nickname = nickname
        self.portrait = get_thumb_path(portrait)

        self.topic_count = convert_to_type_or(topic_count, int, 0)
        self.vote_count = convert_to_type_or(vote_count, int, 0)

        self.membership_level = membership_level
        self.current_city_id = current_city_id
        self.city_id = city_id
        self.city_name = city_name
        self.city_tag_id = convert_to_type_or(city_tag_id, int, None)
        self.province_tag_id = convert_to_type_or(province_tag_id, int, None)
        self.country_tag_id = convert_to_type_or(country_tag_id, int, None)
        self.user_rights_level = user_rights_level
        self.constellation = constellation
        self.membership_icon = membership_icon
        self.level_icon = level_icon
        self.constellation_icon = constellation_icon

    def __bool__(self):
        return self.id is not None

    __nonzero__ = __bool__

    @classmethod
    def get_anonymous_user(cls):
        return cls(None, None, None, None)

    def incr_vote_count(self):
        try:
            self.call_rpc('api/person/incr_vote', user_id=self.id)
            return True

        except RPCFaultException:
            return False

    def decr_vote_count(self):
        try:
            self.call_rpc('api/person/decr_vote', user_id=self.id)
            return True

        except RPCFaultException:
            return False

    def incr_topic_count(self):
        try:
            self.call_rpc('api/person/incr_topic_count', user_id=self.id)
            return True

        except RPCFaultException:
            return False

    def decr_topic_count(self):
        try:
            self.call_rpc('api/person/decr_topic_count', user_id=self.id)
            return True

        except RPCFaultException:
            return False

    @property
    def can_create_video(self):
        from talos.models.topic.videowhitelist import VideoWhiteList

        # 过滤直播黑名单
        try:
            result = self.call_rpc('api/user/is_user_in_blacklist', user_id=self.id)
            if result:
                return False
        except RPCFaultException:
            return False

        if VideoWhiteList.objects.filter(user_id=self.id).exists():
            return True

        # added 7.2.5
        # user rights level > =3,can create video
        if int(self.user_rights_level) >= int(settings.CAN_CREATE_VIDEO_LEVEL):
            return True


        
        # TODO: 产品突然又不要了
        # if user.userextra.is_doctor:
        #     return False

        return False


