# coding=utf8
from __future__ import unicode_literals, absolute_import, print_function

import datetime

from django.db import models
from django.contrib.auth.models import User

from gm_types.doctor import DOCTOR_AREA_TYPE, DOCTOR_LICENCE_TYPE, DOCTOR_INFO_STATUS, DOCTOR_ACCOUNT_TYPE, \
    DOCTOR_ACCOUNT_STATUS
from gm_upload import ImgUrlField, IMG_TYPE

from .tag import Tag
from rpc.cache import doctor_discount_cache

# TODO Skyler 迁移部分
from hippo.models.doctor import Doctor, DoctorEditRecord, HospitalInfo, HospitalLicenes
from hippo.models.doctor import DoctorLicence, DoctorAgreement
from hippo.models.doctor import DoctorAccount, DoctorAccountRecord
from hippo.models.doctor import DoctorCategory
from hippo.models.agreement import Agreement, AgreementDoctor
from hippo.models.extra import DoctorItem, DoctorTag, DocprojectTag
from hippo.models.config import DoctorQuickReplay
from hippo.models.merchant import MerchantRelevance, Merchant


class DocBusinessRecord(models.Model):
    class Meta:
        verbose_name = u'修改医生关联商务记录'
        app_label = 'api'


    doctor = models.ForeignKey(Doctor, verbose_name=u'医生')
    business = models.ForeignKey(User, default=None, null=True,
                                 related_name='doctor_business', verbose_name=u'原关联商务')
    to_business = models.ForeignKey(User, default=None, null=True, related_name='to_business', verbose_name=u'现关联')
    user = models.ForeignKey(User, default=None, null=True, related_name='operate_user', verbose_name=u'操作用户')
    action_time = models.DateTimeField(null=False, auto_now_add=True, verbose_name=u'修改时间')


class DoctorCountRecord(models.Model):
    class Meta:
        verbose_name = u'医生抽成表'
        app_label = 'api'

    doctor = models.ForeignKey(Doctor, verbose_name=u'医生')
    discount = models.IntegerField(default=0, verbose_name=u"总计抽成")

    @classmethod
    def get_diary_index_score(cls, doctor_instance):
        """获取首页日记本的医生抽成分
        有缓存
        """
        try:
            return float(doctor_discount_cache.get('doctor:' + doctor_instance.id))
        except TypeError:
            return float(0.0)


# TODO Skyler DoctorCaseImage DoctorFavor DoctorStat DoctorVote DoctorTagReply 都是需要废弃的
class DoctorCaseImage(models.Model):
    class Meta:
        verbose_name = u'362. 医生案例图片展示'
        verbose_name_plural = u'362. 医生案例图片展示'
        db_table = 'api_doctorcaseimage'
        app_label = 'api'

    doctor = models.ForeignKey(Doctor, related_name="image_doctor", verbose_name=u"医生")
    image = ImgUrlField(img_type=IMG_TYPE.DOCTOR, max_length=255, verbose_name=u"图片URL")
    description = models.TextField(default="", null=True, blank=True,
                                   verbose_name=u"案例说明", help_text=u"最多输入50个汉字")
    im_order = models.IntegerField(default=0, verbose_name=u"展示顺序", help_text=u"小的排在前，大的排在后")

    def __unicode__(self):
        if self.description:
            return self.doctor.name + u"__" + self.description[:10]
        else:
            return self.doctor.name


class DoctorFavor(models.Model):
    class Meta:
        unique_together = ("user", "doctor")
        verbose_name = u'34. 医生收藏'
        verbose_name_plural = u'34. 医生收藏'
        db_table = 'api_doctorfavor'
        app_label = 'api'

    user = models.ForeignKey(User, related_name="doctor_favor_user", help_text=u"用户")
    doctor = models.ForeignKey(Doctor, related_name="doctor_favor_doctor", help_text=u"关注的医生")


class DoctorRegisterInfo(models.Model):
    class Meta:
        verbose_name = u'55. 医生注册申请'
        verbose_name_plural = u'55. 医生注册申请'
        db_table = 'api_doctorregisterinfo'
        app_label = 'api'

    name = models.CharField(max_length=100, verbose_name=u"医生名字")
    phone = models.CharField(max_length=100, verbose_name=u"医生电话")
    hospital = models.CharField(max_length=100, verbose_name=u"医院")
    email = models.EmailField(max_length=100, verbose_name=u"医生邮箱", null=True, blank=True)
    is_handled = models.BooleanField(default=False, verbose_name=u'是否已经处理')
    comments = models.CharField(max_length=200, null=True, blank=True, default='', verbose_name=u'备注')
    created_time = models.DateTimeField(verbose_name=u'申请时间', auto_now_add=True)


class DoctorStat(models.Model):
    class Meta:
        verbose_name = u'医生排序'
        verbose_name_plural = u'医生排序'
        db_table = 'api_doctorstat'
        app_label = 'api'

    doctor = models.ForeignKey(Doctor, related_name='doctorstat')
    doctor_name = models.CharField(max_length=200, verbose_name=u'医生')
    bodypart_id = models.IntegerField()
    bodypart_name = models.CharField(max_length=50, verbose_name=u'部位')
    case_count = models.IntegerField(verbose_name=u'案例数')
    consult_count = models.IntegerField(verbose_name=u'咨询数')
    score_offset = models.IntegerField(default=0, verbose_name=u'分值调整')
    score = models.IntegerField(default=0, verbose_name=u'推荐分值')

    def save(self, *args, **kwargs):
        self.score = self.case_count * 100 + self.consult_count + self.score_offset
        super(DoctorStat, self).save(*args, **kwargs)


class DoctorVote(models.Model):
    class Meta:
        verbose_name = u'361. 医生的点评'
        verbose_name_plural = u'361. 医生的点评'
        db_table = 'api_doctorvote'
        app_label = 'api'

    doctor = models.ForeignKey(Doctor, related_name="voted_doctor", verbose_name=u"医生")
    user = models.ForeignKey(User, related_name="vote_user", verbose_name=u"用户")
    item_prices = models.CharField(max_length=300, default="", null=True, blank=True, verbose_name=u"项目价格")
    content = models.TextField(default="", null=True, blank=True, verbose_name=u"点评内容")
    rate_overall = models.IntegerField(default=0, verbose_name=u"总体评分")
    rate_effect = models.IntegerField(default=0, verbose_name=u"疗效")
    rate_service = models.IntegerField(default=0, verbose_name=u"服务")
    rate_env = models.IntegerField(default=0, verbose_name=u"环境")
    at_hospital = models.BooleanField(default=False, verbose_name=u"是否在医院面诊")
    is_deleted = models.BooleanField(default=False, verbose_name=u"是否删除")
    vote_date = models.DateField(default=datetime.date.today, verbose_name=u"创建时间")

    def __unicode__(self):
        return self.item_prices


class DoctorTagReply(models.Model):
    class Meta:
        app_label = 'api'

    doctor = models.ForeignKey(Doctor)
    tag = models.ForeignKey(Tag)
    reply_num = models.IntegerField(default=0, verbose_name=u'医生回复帖子数量')
