# -*- coding:utf-8 -*-
'''
date: 2019-04-04
用户地推活动, 提供临时接口, 基本上一次性使用
'''

import datetime

from gm_types.gaia import TEMP_ACTIVITY_TYPE, EXCHANGE_TYPE
from rpc.decorators import bind_context, bind
from api.tool.user_tool import get_user_from_context

from api.models.user_activity import UserShareCode, UserShowInfo, UserInvite, ThirdRecord, UserExchange

REG = 1
LOGIN = 2
FINISH = 3

aty_type = code_type = TEMP_ACTIVITY_TYPE.GROUND_PUSH1904
SUC_INV_COUNT = 6

@bind('account/activity/ground_push/reg_callback')
def reg_callback(user_id, share_code, aty_openid):
    '''
    被邀请者注册后:
        a. 记录其对应的微信信息, 在活动中做展示
        b. 记录其邀请码信息, 关联邀请者-被邀请者
    user_id: 注册用户id
    share_code: 邀请码
    aty_openid: 微信openid
    '''
    if aty_openid:
        UserShowInfo.create_showinfo(user_id, aty_type, aty_openid)
    if not share_code:
        return
    user_code_obj = UserShareCode.get_obj_by_share_code_type(share_code, code_type)
    if user_code_obj:
        now_time = datetime.datetime.now()
        inviter_id = user_code_obj.user_id
        invite_status = REG if user_code_obj.end_time >= now_time else FINISH
        UserInvite.create_relation(inviter_id, user_id, invite_status, aty_type, share_code)


@bind('account/activity/ground_push/login_callback')
def login_callback(user_id, aty_openid, from_app=False):
    '''
    被邀请者在app端登录后, 完成邀请闭环
    user_id:
    '''
    if aty_openid:
        UserShowInfo.create_showinfo(user_id, aty_type, aty_openid)

    if not from_app:
        return

    inv_obj = UserInvite.get_record_by_invitee_aty(user_id, aty_type)
    if not inv_obj:
        return

    if inv_obj.invite_status in (LOGIN, FINISH):
        return

    if inv_obj.invite_status == REG:
        now_time = datetime.datetime.now()
        user_code_obj = UserShareCode.get_obj_by_share_code_type(inv_obj.share_code, inv_obj.activity_type)
        invite_status = LOGIN if user_code_obj.end_time >= now_time else FINISH
        inv_obj.invite_status = invite_status
        inv_obj.save()


@bind_context('account/activity/ground_push/share_data')
def get_share_data(ctx):
    '''
    获取用户的分享数据
    '''
    user = get_user_from_context(ctx)
    if not user:
        return {}
    user_id = user.id
    invitees = UserInvite.get_inviteeinfo_by_inviter_aty_status(user_id, aty_type, LOGIN)
    share_code_info = UserShareCode.get_sharecode_by_user_aty(user_id, aty_type)
    now = datetime.datetime.now()
    end_time = share_code_info.pop('end_time', None) or now
    countdown = (end_time - now).total_seconds()
    result = {}
    result['user_id'] = user_id
    result['open_id'] = UserShowInfo.get_platformid_by_userid(user_id, aty_type)
    result['invitee_info'] = invitees
    result['countdown'] = 0 if countdown <= 0 else countdown
    result['share_code'] = share_code_info['share_code']
    result['has_exchange'] = bool(UserExchange.get_record_by_uid_aty(user_id, aty_type))
    return result


@bind('account/activity/ground_push/record_wechat_info')
def record_wechat_info(openid, nickname, imgurl, w_city, **kwargs):
    '''
    记录微信code对应的openid数据
    '''
    ThirdRecord.create_info_record(openid, nickname, imgurl, w_city, aty_type)


@bind_context('account/activity/ground_push/exchange_gift')
def exchange_gift(ctx, user_id=None):
    user = get_user_from_context(ctx)
    if not user:
        return {'success':0, 'message': '账号异常'}
    user_id = user.id
    invitees = UserInvite.get_inviteeinfo_by_inviter_aty_status(user_id, aty_type, LOGIN)
    if len(invitees) < SUC_INV_COUNT:
        return {'success': 0, 'message': '未达到领取条件'}
    record = UserExchange.get_record_by_uid_aty(user_id, aty_type)
    if record:
        return {'success':0, 'message': '已经兑换'}

    UserExchange.create_record(user_id, EXCHANGE_TYPE.APPLY, aty_type)
    return {'success':1, 'message': '兑换成功'}
