#!/usr/bin/env python
# coding=utf8

from __future__ import absolute_import

import datetime

from django.db import models

from gm_types.doctor import DOCTOR_ACTIVITY_STATUS

from .doctor import Doctor


class DoctorActivity(models.Model):
    """ 医生端活动 """
    class Meta:
        app_label = 'doctor'
        verbose_name = u'医生端活动列表'

    content = models.OneToOneField('Content', related_name='doctor_activity', verbose_name=u'医生运营内容')
    title = models.CharField(u'标题', max_length=100)
    is_online = models.BooleanField(u'是否上线', default=False)
    begin_time = models.DateTimeField(u'开始时间')
    end_time = models.DateTimeField(u'结束时间')
    created_time = models.DateTimeField(u'创建时间', auto_now_add=True)
    update_time = models.DateTimeField(u'最后更新时间', auto_now=True)

    def update_req_data(self, req_data):
        self.content.title = req_data['title']
        self.content.content = req_data['content_data']
        self.title = req_data['title']
        self.is_online = req_data['is_online']
        self.begin_time = req_data['begin_time']
        self.end_time = req_data['end_time']
        self.update_time = datetime.datetime.now()
        self.content.save()
        self.save()

    def status(self, doctor):
        """
            活动的状态
        """
        if not self.is_online:
            return DOCTOR_ACTIVITY_STATUS.DEFAULT
        now_datetime = datetime.datetime.now()
        if now_datetime < self.begin_time:
            return DOCTOR_ACTIVITY_STATUS.WILL_START
        if now_datetime > self.end_time:
            return DOCTOR_ACTIVITY_STATUS.OVER
        if doctor:
            try:
                ActivityParticipate.objects.get(activity=self, doctor=doctor)
                return DOCTOR_ACTIVITY_STATUS.HAVE_APPLIED
            except ActivityParticipate.DoesNotExist:
                pass
        return DOCTOR_ACTIVITY_STATUS.PROCESSING


class ActivityParticipate(models.Model):
    """ 活动参加者 """
    class Meta:
        app_label = 'doctor'
        verbose_name = u'医生端活动参加列表'

    activity = models.ForeignKey(DoctorActivity, verbose_name=u"医生端活动")
    doctor = models.ForeignKey(Doctor, verbose_name=u'参加活动的医生')
    created_time = models.DateTimeField(u'报名时间', auto_now_add=True)
