# coding=utf-8

from django.db import models
from django.utils.http import urlquote
from gm_upload import IMG_TYPE, ImgUrlField

from gm_types.gaia import (
    SLIDE_PAYMENT_TYPE,
    SLIDE_USER_TYPE,
    SERVICE_HOME_OPERATION,
    SERVICE_DISPLAY_PORT,
    FONT_COLOR,
    SERVICEHOME_OPERATION_POSITION,
    SLIDE_TYPE
)

from api.models import (
    SERVICE_ACTIVITY_TYPE,
    SERVICE_CHANNEL,
    Special,
    Service)
from api.models import Region, City, Tag

from rpc.tool.protocol import PushUrlProtocol


class ServiceActivity(models.Model):
    class Meta:
        verbose_name = u'美购活动'
        verbose_name_plural = u'美购活动'
        db_table = 'api_service_activity'
        app_label = 'api'

    TOP_RIGHT_POSITION = 2
    BOTTOM_RIGHT_POSITION = 3

    name = models.CharField(max_length=200, null=False, verbose_name=u'活动名称')
    activity_type = models.CharField(max_length=1, null=False, verbose_name=u'活动类型', choices=SERVICE_ACTIVITY_TYPE)
    special = models.ForeignKey(Special, null=True, verbose_name=u'关联专题')
    position = models.IntegerField(default=1, verbose_name=u'位置')
    image = ImgUrlField(img_type=IMG_TYPE.SERVICEACTIVITY, max_length=200, null=True, verbose_name=u'图片')
    update_time = models.DateTimeField(auto_now=True, verbose_name=u'最后更新时间')

    @classmethod
    def left(cls):
        """fixed as famous doctor."""
        ac = cls.objects.get(position=1)
        return {
            'url': PushUrlProtocol.SERVICE_LIST.format(
                channel_id=SERVICE_CHANNEL.FAMOUS_DOCTOR,
                section_id='all',
                title=urlquote(ac.name),
            ),
            'image': ac.image,
            'type': SERVICE_ACTIVITY_TYPE.FAMOUS_DOCTORS,
        }

    @classmethod
    def _get_top_right_and_bottom_right(cls, position):
        ac = cls.objects.get(position=position)
        return {
            'url': PushUrlProtocol.PROMOTION_SPECIAL.format(id=ac.special.id),
            'image': ac.image,
            'type': SERVICE_ACTIVITY_TYPE.SPECIAL,
        }

    @classmethod
    def top_right(cls):
        """fixed as free activities entry."""
        return cls._get_top_right_and_bottom_right(cls.TOP_RIGHT_POSITION)

    @classmethod
    def bottom_right(cls):
        """fixed as service special list."""
        return cls._get_top_right_and_bottom_right(cls.BOTTOM_RIGHT_POSITION)


class ServiceRecommend(models.Model):
    class Meta:
        verbose_name = u'美购推荐'
        db_table = 'api_servicerecommend'
        app_label = 'api'

    service = models.OneToOneField(Service, verbose_name=u'美购', related_name='recommend')
    is_recommend = models.BooleanField(verbose_name=u'是否推荐', default=False, null=False)
    rank = models.IntegerField(verbose_name=u'排序', default=999999, null=False)
    is_delete = models.BooleanField(verbose_name=u'是否删除', default=False, null=False)


class NewCategory(models.Model):
    class Meta:
        verbose_name = u'新品类模块'
        db_table = 'api_new_category'
        app_label = 'api'

    icon = ImgUrlField(u'品类icon', max_length=255, null=False, default='', img_type=IMG_TYPE.SERVICEHOME)
    name = models.CharField(u'名称', max_length=128, null=False, default='')
    ordering = models.IntegerField(u'顺序', null=False, default=0)
    tags = models.ManyToManyField(Tag, verbose_name=u'关联tag')
    regions = models.ManyToManyField(Region, verbose_name=u'展示大区')
    cities = models.ManyToManyField(City, verbose_name=u'展示城市')
    user_type = models.CharField(u'用户类型', max_length=2, choices=SLIDE_USER_TYPE.choices, default=SLIDE_USER_TYPE.ALL_USER, null=False)
    payment_type = models.CharField(u'有无支付', max_length=2, choices=SLIDE_PAYMENT_TYPE.choices, default=SLIDE_PAYMENT_TYPE.ALL_PAYMENT, null=False)
    is_online = models.BooleanField(u"是否上线", null=False, default=False)
    start_time = models.DateTimeField(u'开始时间')
    end_time = models.DateTimeField(u'结束时间')
    create_time = models.DateTimeField(u'创建时间', auto_now_add=True)
    update_time = models.DateTimeField(u'更新时间', auto_now=True)


class ServiceHomeBackground(models.Model):
    class Meta:
        verbose_name = u'美购首页背景'
        db_table = 'api_service_home_background'
        app_label = 'api'

    start_time = models.DateTimeField(u'开始时间')
    end_time = models.DateTimeField(u'结束时间')
    background_img = ImgUrlField(u'背景图', max_length=255, null=False, default='', img_type=IMG_TYPE.SERVICEHOME)
    font_color = models.CharField(u'字体颜色', max_length=2, choices=FONT_COLOR.choices, default=FONT_COLOR.BLACK, null=False)
    is_online = models.BooleanField(u"是否上线", null=False, default=False)
    create_time = models.DateTimeField(u'创建时间', auto_now_add=True)
    update_time = models.DateTimeField(u'更新时间', auto_now=True)


class ServiceHomeOperation(models.Model):
    class Meta:
        verbose_name = u'美购首页运营位'
        db_table = 'api_service_home_operation'
        app_label = 'api'

    operation_type = models.CharField(u'运营位类型', max_length=2, choices=SERVICE_HOME_OPERATION.choices, default=SERVICE_HOME_OPERATION.ARRANGE, null=False)
    start_time = models.DateTimeField(u'开始时间')
    end_time = models.DateTimeField(u'结束时间')
    user_type = models.CharField(u'用户类型', max_length=2, choices=SLIDE_USER_TYPE.choices, default=SLIDE_USER_TYPE.ALL_USER, null=False)
    payment_type = models.CharField(u'有无支付', max_length=2, choices=SLIDE_PAYMENT_TYPE.choices, default=SLIDE_PAYMENT_TYPE.ALL_PAYMENT, null=False)
    is_online = models.BooleanField(u"是否上线", null=False, default=False)
    display_port = models.IntegerField(u'展示端口', null=False, default=0)
    regions = models.ManyToManyField(Region, verbose_name=u'展示大区')
    cities = models.ManyToManyField(City, verbose_name=u'展示城市')
    details = models.TextField(u'运营位配置', null=True)
    create_time = models.DateTimeField(u'创建时间', auto_now_add=True)
    update_time = models.DateTimeField(u'更新时间', auto_now=True)

    @property
    def display_desc(self):
        display_lenth = len(SERVICE_DISPLAY_PORT.choices)
        return [SERVICE_DISPLAY_PORT.getDesc(str(i)) for i in range(display_lenth, 0, -1) if (self.display_port % 2 ** (i+1)) // 2 ** i]


class ServiceRelatedOperation(models.Model):
    class Meta:
        verbose_name = u'美购首页运营位关联配置'
        db_table = 'api_servicerelatedoperation'
        unique_together = ('servicehome', 'position')
        app_label = 'api'

    image = ImgUrlField(u'运营位图片', max_length=128, default='', img_type=IMG_TYPE.SERVICEHOME)
    jump_id = models.CharField(u'跳转', max_length=128, default='')
    jump_type = models.CharField(max_length=3, choices=SLIDE_TYPE, default=SLIDE_TYPE.SPECIAL, verbose_name=u'跳转到')
    servicehome = models.ForeignKey(ServiceHomeOperation, verbose_name=u'关联运营位',
                                    related_name=u'servicerelatedoperations')
    position = models.SmallIntegerField(verbose_name=u'位置', default=SERVICEHOME_OPERATION_POSITION.FIRST)
    create_time = models.DateTimeField(u'创建时间', auto_now_add=True)
    update_time = models.DateTimeField(u'更新时间', auto_now=True)