# coding=utf8
from __future__ import unicode_literals, absolute_import, print_function

from django.db import models

from gm_types.gaia import CITY_LEVEL

from rpc.tool.dict_mixin import to_dict
from .tag import Tag


class Area(models.Model):
    class Meta:
        verbose_name = u'地区'
        verbose_name_plural = u'地区'
        db_table = 'api_area'
        app_label = 'api'

    area_name = models.CharField(max_length=20, verbose_name=u'地区名字')
    phone_prefix = models.CharField(max_length=5, verbose_name=u'地区号码前缀')

    def data(self):
        return {
            "id": self.id,
            "name": self.area_name,
            "prefix": self.phone_prefix
        }


class Country(models.Model):
    class Meta:
        verbose_name = u'国家'
        verbose_name_plural = u'国家'
        db_table = 'api_country'
        app_label = 'api'

    id = models.CharField(max_length=40, primary_key=True, verbose_name=u'国家英文名')
    name = models.CharField(max_length=40, null=False, verbose_name=u'国家中文名')
    tag = models.ForeignKey(Tag, null=True, blank=True)

    def __unicode__(self):
        return self.name

    def data(self):
        return {
            'name': self.name,
            'id': self.id,
            'tag_id': self.tag.id,
            'provinces': [province.province_data()
                          for province in self.province_set.all()]
        }


class Region(models.Model):
    class Meta:
        verbose_name = u'大区'
        app_label = 'api'

    english_name = models.CharField(max_length=40, verbose_name=u'大区英文名')
    name = models.CharField(max_length=40, verbose_name=u'大区中文名')

    def cities(self):
        return City.objects.filter(province__in=self.province_set.all())


class Province(models.Model):
    class Meta:
        verbose_name = u'省份'
        verbose_name_plural = u'省份'
        db_table = 'api_province'
        app_label = 'api'

    id = models.CharField(max_length=40, primary_key=True, verbose_name=u'省份拼音名')
    name = models.CharField(max_length=20, null=False, verbose_name=u'中文名')
    country = models.ForeignKey(Country, null=True, verbose_name=u'国家')
    region = models.ForeignKey(Region, null=True, blank=True, verbose_name=u'大区')
    display_in_filter = models.BooleanField(default=True, verbose_name=u'显示在筛选中')
    tag = models.ForeignKey(Tag, null=True, blank=True)
    is_municipality = models.BooleanField(default=False,
                                          verbose_name=u"是否为直辖市(True:是,False:否)")

    def province_data(self):
        result = dict()
        result['name'], result['id'], result['tag_id'] = self.name, self.id, self.tag_id
        # 检查当前省份是否是直辖市
        result['is_municipality'] = self.is_municipality
        cities = list()
        for city in self.cities.all():
            cities.append(city.city_data())
        result['cities'] = cities

        return result

    def cities(self):
        return City.objects.filter(province=self)

    def __unicode__(self):
        return self.name


class OnlineCityManager(models.Manager):
    """only return online city."""

    def get_queryset(self):
        return super(OnlineCityManager, self).get_queryset().filter(
            is_online=True
        )


class City(models.Model):

    class Meta:
        verbose_name = u'62. 城市'
        verbose_name_plural = u'62. 城市'
        db_table = 'api_city'
        app_label = 'api'

    id = models.CharField(max_length=40, primary_key=True, verbose_name=u'拼音名')
    name = models.CharField(max_length=20, null=False, verbose_name=u'中文名')
    province = models.ForeignKey(Province, null=True, related_name='cities', verbose_name=u'省份')
    display_in_filter = models.BooleanField(default=True, verbose_name=u'显示在筛选中')
    is_online = models.BooleanField(default=True, verbose_name=u'是否在线')
    tag = models.ForeignKey(Tag)
    level = models.PositiveIntegerField('城市级别', default=CITY_LEVEL.FOUR)

    objects = OnlineCityManager()

    def city_data(self):
        return {'name': self.name, 'id': self.id, 'tag_id': self.tag_id}

    def is_hospital_exists(self):
        return self.hospital_set.exists()

    is_hospital_exists.boolean = True
    is_hospital_exists.short_description = u'是否有医院'

    def hospitals(self):
        return self.hospital_set.all()

    def to_dict(self):
        result = to_dict(self)
        return result

    def __unicode__(self):
        return self.name

